/*!
 * Ultra DOM Tools – in-page visual DOM editor (STRICT scoped to #pagePreview iframe)
 * Gate conditions (all must pass):
 *  1) running inside an iframe (not top window)
 *  2) iframe name === 'ua_preview'
 *  3) URL has ?ua_preview=1 (or ?ua_preview)
 *  4) frameElement.parentElement.id === 'pagePreview'
 *  5) parent sends postMessage({type:'UA_DOMTOOLS_ENABLE', nonce})
 */

(function(){
  // منع التكرار
  if (window.__UA_DOMTOOLS_BOOTED__) return;
  window.__UA_DOMTOOLS_BOOTED__ = true;

  // لازم نكون داخل iframe
  try { if (window.top === window) return; } catch(_) { return; }

  // لازم اسم الإطار وبارامتر المعاينة
  var qs = new URLSearchParams(location.search || '');
  if (!(qs.has('ua_preview'))) return;
  if (window.name !== 'ua_preview') return;

  // لازم فعلاً يكون الإطار داخل #pagePreview (من ناحية الأب)
  var okByPlacement = false;
  try {
    var fe = window.frameElement;
    okByPlacement = !!(fe && fe.parentElement && fe.parentElement.id === 'pagePreview');
  } catch(_) {}
  if (!okByPlacement) return;

  // هنستنى إذن صريح من الأب (Handshake)
  var allowedByParent = false;
  var parentNonce = null;

  // ======= mount() will build the UI ONLY after handshake passes =======
  function mount(){
    if (allowedByParent !== true || !parentNonce) return;
    if (window.__UA_DOMTOOLS_MOUNTED__) return;
    window.__UA_DOMTOOLS_MOUNTED__ = true;

    /* ---------------------------------- UI ---------------------------------- */
    const ui = document.createElement('div');
    ui.id = 'uaDomToolsBar';
    ui.className = 'ua-domtools';
    ui.setAttribute('data-ua-domtools','');

    ui.style.cssText = [
      'position:fixed','top:12px','right:12px','z-index:2147483647',
      'background:#0b1220cc','color:#e2e8f0','border:1px solid #1f2a44',
      'border-radius:12px','padding:10px','font:12px system-ui',
      'backdrop-filter:blur(8px)','-webkit-backdrop-filter:blur(8px)'
    ].join(';');

    function btn(act, label, danger){
      return `<button data-act="${act}" class="dt-btn${danger? ' dt-danger':''}">${label}</button>`;
    }

    ui.innerHTML =
      '<div style="display:flex;gap:6px;flex-wrap:wrap;align-items:center">'+
        '<strong style="color:#7dd3fc">DOM Tools</strong>'+
        btn('pick','Pick')+btn('up','Up')+btn('down','Down')+
        btn('wrap','Wrap')+btn('unwrap','Unwrap')+
        btn('clone','Clone')+btn('delete','Delete',true)+
        btn('html','Edit&nbsp;HTML')+btn('attrs','Attrs')+btn('style','CSS/JS')+
        '<span style="flex:1"></span>'+
        '<button data-act="save" class="dt-btn" style="background:#10b981;color:#04110d;border-radius:8px;padding:4px 10px"><b>Save</b></button>'+
      '</div>'+
      '<div id="dt-target" style="margin-top:8px;color:#94a3b8;max-width:420px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis">(no selection)</div>'+
      '<div style="margin-top:6px;color:#6b7280">Shortcuts: P, ↑/↓, W, U, C, Del, H, A, S, Esc</div>';

    const baseCss =
`.dt-btn{background:#111827;border:1px solid #334155;color:#e2e8f0;padding:4px 8px;border-radius:6px;cursor:pointer}
.dt-btn:hover{background:#1f2937}
.dt-danger{color:#fca5a5}
#uaDomToolsBar .kbd{padding:1px 4px;border:1px solid #334155;border-radius:4px;background:#0b1220;color:#9ca3af}
#uaDomToolsBar .row{display:flex;gap:8px;align-items:center;flex-wrap:wrap}
#uaDomToolsBar .muted{color:#94a3b8}
#ua-hilite-box{position:fixed;border:2px solid #22d3ee;background:rgba(34,211,238,.08);z-index:2147483646;pointer-events:none}
[data-ua-picking="1"] #uaDomToolsBar{display:none !important} /* إخفاء محلي وقت الالتقاط */
`;
    const st = document.createElement('style'); st.textContent = baseCss;
    document.head.appendChild(st);
    document.documentElement.appendChild(ui);

    /* ------------------------------ State/Helpers ---------------------------- */
    let sel = null, picking = false, box = null, lastScroll = {x: 0, y: 0};

    function ensureBox(){
      if (!box) {
        box = document.createElement('div');
        box.id = 'ua-hilite-box';
        document.body.appendChild(box);
      }
      return box;
    }
    function hilite(el){
      if (!el || !el.getBoundingClientRect) return;
      const r = el.getBoundingClientRect();
      const b = ensureBox();
      b.style.left = r.left + 'px';
      b.style.top = r.top + 'px';
      b.style.width = r.width + 'px';
      b.style.height = r.height + 'px';
    }
    function elPath(el){
      if (!el || el.nodeType !== 1) return '';
      if (el.id) return `${el.tagName.toLowerCase()}#${el.id}`;
      const seg = [];
      let e = el;
      while (e && e.nodeType === 1) {
        let k = 1, s = e;
        while ((s = s.previousElementSibling)) if (s.tagName === e.tagName) k++;
        seg.unshift(`${e.tagName.toLowerCase()}:nth-of-type(${k})`);
        if (e.tagName.toLowerCase() === 'html') break;
        e = e.parentElement;
      }
      return seg.join(' > ');
    }
    function setSel(el){
      sel = el || null;
      const label = sel ? elPath(sel) : '(no selection)';
      const tgt = document.getElementById('dt-target'); if (tgt) tgt.textContent = label;
      if (sel) hilite(sel);
    }
    function isUi(el){ return el === ui || ui.contains(el) || (box && (el === box || box.contains(el))); }

    /* ------------------------------- Picking -------------------------------- */
    function startPick(){
      lastScroll = {x: window.scrollX, y: window.scrollY};
      picking = true;
      document.documentElement.setAttribute('data-ua-picking','1');
      alert('اختيار عنصر: حرّك الفأرة ثم انقر على العنصر المطلوب. (Esc للإلغاء)');
    }
    function endPick(){
      picking = false;
      document.documentElement.removeAttribute('data-ua-picking');
      if (box) box.style.width = box.style.height = '0px';
    }
    function cancelPick(){ endPick(); }

    document.addEventListener('mousemove', function(e){
      if (!picking) return;
      const t = e.target;
      if (!t || isUi(t)) return;
      hilite(t);
    }, true);

    document.addEventListener('click', function(e){
      if (!picking) return;
      const t = e.target;
      if (isUi(t)) return; // تجاهل نقر الشريط نفسه
      e.preventDefault();
      e.stopPropagation();
      setTimeout(()=>window.scrollTo(lastScroll.x, lastScroll.y), 0);
      endPick();
      setSel(t);
      alert('✔️ تم اختيار العنصر: ' + elPath(t));
    }, true);

    document.addEventListener('keydown', function(e){
      const k = e.key.toLowerCase();
      if (k === 'escape' && picking) { cancelPick(); e.preventDefault(); }
    }, true);

    /* -------------------------------- Actions -------------------------------- */
    function moveUp(){
      if (!sel || !sel.previousElementSibling) return;
      sel.parentNode.insertBefore(sel, sel.previousElementSibling);
      hilite(sel);
    }
    function moveDown(){
      if (!sel || !sel.nextElementSibling) return;
      sel.parentNode.insertBefore(sel.nextElementSibling, sel);
      hilite(sel);
    }
    function doWrap(){
      if (!sel) return;
      const w = document.createElement('div');
      w.className = 'ua-wrap';
      sel.parentNode.insertBefore(w, sel);
      w.appendChild(sel);
      setSel(w);
    }
    function doUnwrap(){
      if (!sel || !sel.parentElement) return;
      const p = sel.parentElement;
      if (p === document.body || p === document.documentElement) return;
      while (sel.firstChild) p.insertBefore(sel.firstChild, sel);
      sel.remove();
      setSel(p);
    }
    function doClone(){
      if (!sel) return;
      const c = sel.cloneNode(true);
      sel.parentNode.insertBefore(c, sel.nextSibling);
      setSel(c);
    }
    function doDelete(){
      if (!sel) return;
      const p = sel.parentElement;
      sel.remove();
      setSel(p);
    }
    function editHTML(){
      if (!sel) return;
      const cur = sel.innerHTML;
      const nv = prompt('Edit innerHTML:', cur);
      if (nv === null) return;
      sel.innerHTML = nv;
      hilite(sel);
    }
    function editAttrs(){
      if (!sel) return;
      const pairs = [].slice.call(sel.attributes)
        .map(a => a.name+'="'+a.value.replace(/"/g,'&quot;')+'"')
        .join(' ');
      const nv = prompt('Attributes (name="value" ...):', pairs);
      if (nv === null) return;
      [].slice.call(sel.attributes).forEach(a => sel.removeAttribute(a.name)); // wipe
      nv.trim().split(/\s+(?=\S+=")/).forEach(ch => {
        const m = ch.match(/^([\w:-]+)="(.*)"$/);
        if (m) sel.setAttribute(m[1], m[2].replace(/&quot;/g,'"'));
      });
      hilite(sel);
    }
    function editStyleJs(){
      if (!sel) return;
      const css = prompt('Custom CSS for this element (inline style):', sel.getAttribute('style')||'');
      if (css !== null) sel.setAttribute('style', css);
      const js = prompt('Run element JS (optional, executed now – el is available):', '');
      if (js) {
        try { (new Function('el', js))(sel); }
        catch(err){ alert('JS error: '+err); }
      }
      hilite(sel);
    }

    /* --------------------------------- Save ---------------------------------- */
    function serializeFullHTML(){
      const doctype = document.doctype
        ? `<!doctype ${document.doctype.name}${
            document.doctype.publicId ? ` PUBLIC "${document.doctype.publicId}"` : ''
          }${document.doctype.systemId ? ` "${document.doctype.systemId}"` : ''}>`
        : '<!doctype html>';
      return doctype + '\n' + document.documentElement.outerHTML;
    }
    function doSave(){
      const html = serializeFullHTML();
      // نبعث للأب مع نفس الـnonce للتحقق
      window.parent.postMessage({ type:'UA_DOM_SAVE', html, nonce: parentNonce }, '*');
      alert('🔒 Snapshot جاهز — سيتم حفظه من لوحة الإدارة.');
    }

    /* --------------------------- Keyboard shortcuts -------------------------- */
    document.addEventListener('keydown', function(e){
      if (e.ctrlKey || e.metaKey || e.altKey) return;
      const k = e.key.toLowerCase();
      if (k === 'p') { startPick(); e.preventDefault(); }
      else if (k === 'arrowup') { moveUp(); e.preventDefault(); }
      else if (k === 'arrowdown') { moveDown(); e.preventDefault(); }
      else if (k === 'w') { doWrap(); e.preventDefault(); }
      else if (k === 'u') { doUnwrap(); e.preventDefault(); }
      else if (k === 'c') { doClone(); e.preventDefault(); }
      else if (k === 'delete' || k === 'backspace') { doDelete(); e.preventDefault(); }
      else if (k === 'h') { editHTML(); e.preventDefault(); }
      else if (k === 'a') { editAttrs(); e.preventDefault(); }
      else if (k === 's') { doSave(); e.preventDefault(); }
    }, true);

    /* ------------------------------- Wire buttons ---------------------------- */
    ui.addEventListener('click', function(e){
      const b = e.target.closest('.dt-btn'); if (!b) return;
      const act = b.getAttribute('data-act');
      if (act === 'pick') startPick();
      else if (act === 'up') moveUp();
      else if (act === 'down') moveDown();
      else if (act === 'wrap') doWrap();
      else if (act === 'unwrap') doUnwrap();
      else if (act === 'clone') doClone();
      else if (act === 'delete') doDelete();
      else if (act === 'html') editHTML();
      else if (act === 'attrs') editAttrs();
      else if (act === 'style') editStyleJs();
      else if (act === 'save') doSave();
    });

    // أثناء الالتقاط: لا تنقلك الروابط
    document.addEventListener('click', function(e){
      if (!picking) return;
      const a = e.target.closest('a[href]');
      if (a) { e.preventDefault(); e.stopImmediatePropagation(); }
    }, true);

    // لو حصل blur أثناء الاختيار، رجّع نفس موضع التمرير
    window.addEventListener('blur', function(){
      if (picking) setTimeout(()=>window.scrollTo(lastScroll.x, lastScroll.y), 0);
    });

    // Expose minimal API
    window.UADomTools = { select: setSel, save: doSave, pick: startPick };
  } // end mount

  // ======= Handshake: لا تفعيل بدون إذن من الأب =======
  window.addEventListener('message', function(e){
    if (!e || !e.data) return;
    if (e.data.type === 'UA_DOMTOOLS_ENABLE' && e.data.nonce) {
      allowedByParent = true;
      parentNonce = String(e.data.nonce);
      mount();
    }
  });
})();
