// polling.js — Auto-advance by admin order (settings.json) OR by p param
// Options:
//   ?polldelay=3000   -> delay ms (default 3000)
//   ?pollloop=1       -> when reach last page, wrap to first
//   ?polldebug=1      -> verbose console logs
// Works with:
//   - index.php?sid=...&p=1 (router, 1-based)
//   - page_xxxxxxxx.php (standalone) -> converts to router with matching p

(function () {
  try {
    // --- Guard: ignore admin preview ---
    if (/\bpv=\d+/.test(location.search)) return;

    const url = new URL(location.href);
    const qs = url.searchParams;
    const DEBUG = qs.get('polldebug') === '1';

    const log = (...a) => { if (DEBUG) try { console.log('[polling]', ...a); } catch (_) {} };

    const toInt = (v, d = null) => {
      const n = parseInt(v, 10);
      return Number.isFinite(n) ? n : d;
    };

    // Delay config (query > data-attr > default)
    const delay =
      toInt(qs.get('polldelay'), null) ??
      toInt(document.body?.dataset?.polldelay || '', null) ??
      3000;

    const LOOP = qs.get('pollloop') === '1';

    // Ensure we have a sid (use existing or generate lightweight one)
    let sid = qs.get('sid') || '';
    if (!sid) {
      try {
        sid = sessionStorage.getItem('ua_sid');
        if (!sid) {
          sid = Date.now().toString(36) + Math.random().toString(36).slice(2, 10);
          sessionStorage.setItem('ua_sid', sid);
        }
      } catch (_) {}
    }

    // Resolve base path to index.php (same directory)
    const baseDir = location.pathname.replace(/[^/]+$/, '');
    const indexHref = location.origin + baseDir + 'index.php';

    // Determine current pointer (p) or index by file name
    const pNow = toInt(qs.get('p'), null);
    const currentFile = location.pathname.split('/').pop();

    // Prevent infinite bounces: if we tried redirecting here too many times, stop
    // We store "<path>?<p|file>" key and count attempts
    const tryKey = (() => {
      const keyPart = pNow ? 'p=' + pNow : 'file=' + currentFile;
      return 'poll_try_' + location.origin + baseDir + keyPart;
    })();
    const tries = toInt(sessionStorage.getItem(tryKey), 0) || 0;
    if (tries > 5) {
      log('Abort: too many attempts for', tryKey);
      return;
    }

    // Helper: safe redirect
    function goNext(targetUrl) {
      try {
        sessionStorage.setItem(tryKey, String(tries + 1));
      } catch (_) {}
      log('Redirecting after', delay, 'ms to', targetUrl);
      setTimeout(() => { location.href = targetUrl; }, delay);
    }

    // Try to advance by p directly (fast path), optionally bounded by settings.json
    async function advanceByP(p) {
      let total = null;
      try {
        const res = await fetch(baseDir + 'settings.json?' + Date.now(), { cache: 'no-store' });
        const st = await res.json().catch(() => null);
        const pages = Array.isArray(st?.pages) ? st.pages : [];
        total = pages.length || null;
        log('settings loaded, total pages =', total);
      } catch (_) {
        log('settings fetch failed — will fall back to blind advance');
      }

      let nextP = p + 1;

      // If we have total, bound nextP to [1..total] or loop if requested
      if (typeof total === 'number' && total > 0) {
        if (nextP > total) {
          if (LOOP) {
            nextP = 1;
            log('loop enabled -> wrapping to p=1');
          } else {
            log('reached last page, stopping');
            return;
          }
        }
      }

      const next = new URL(indexHref);
      if (sid) next.searchParams.set('sid', sid);
      next.searchParams.set('p', String(nextP));
      // keep hash
      if (url.hash) next.hash = url.hash;

      goNext(next.toString());
    }

    // Try to map current standalone page to its p via settings.json
    async function advanceByFilename(file) {
      try {
        const res = await fetch(baseDir + 'settings.json?' + Date.now(), { cache: 'no-store' });
        const st = await res.json().catch(() => null);
        const pages = Array.isArray(st?.pages) ? st.pages : [];
        if (!pages.length) {
          log('no pages in settings; cannot map filename -> p');
          return;
        }
        const idx = pages.findIndex(p => (p?.php || '').toLowerCase() === file.toLowerCase());
        if (idx === -1) {
          log('current file not found in settings:', file);
          return;
        }
        const total = pages.length;
        let nextIdx = idx + 1;
        if (nextIdx >= total) {
          if (LOOP) nextIdx = 0;
          else {
            log('at last file, stopping (no loop)');
            return;
          }
        }

        const next = new URL(indexHref);
        if (sid) next.searchParams.set('sid', sid);
        next.searchParams.set('p', String(nextIdx + 1));
        if (url.hash) next.hash = url.hash;

        goNext(next.toString());
      } catch (e) {
        log('settings fetch/match failed for filename', e);
      }
    }

    // ENTRY
    if (pNow && pNow > 0) {
      log('mode=router by p', pNow);
      advanceByP(pNow);
      return;
    }

    // If not router mode, try filename mode
    if (/^page_[a-z0-9]{8}\.php$/i.test(currentFile)) {
      log('mode=standalone by filename', currentFile);
      advanceByFilename(currentFile);
      return;
    }

    // If neither, do nothing (probably admin or non-flow page)
    log('not a flow page — idle');
  } catch (e) {
    // Silent fail
    // console.error('[polling] fatal', e);
  }
})();
